<?php

namespace Drupal\htmx_extras\Plugin\views\row;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\RevisionableInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Theme\ComponentPluginManager;
use Drupal\htmx_extras\HtmxExtrasHelper;
use Drupal\htmx_extras\Render\HtmxEntityPartial;
use Drupal\search_api\IndexInterface;
use Drupal\search_api\Plugin\views\query\SearchApiQuery;
use Drupal\search_api\SearchApiException;
use Drupal\search_api\Utility\Utility;
use Drupal\views\Attribute\ViewsRow;
use Drupal\views\Plugin\views\display\DisplayPluginBase;
use Drupal\views\Plugin\views\row\RowPluginBase;
use Drupal\views\ViewExecutable;
use Symfony\Component\DependencyInjection\ContainerInterface;

#[ViewsRow(
  id: 'search_api_htmx_row',
  title: new TranslatableMarkup('Search API HTMX Row'),
  display_types: ['normal'],
  theme: 'views_view_fields'
)]
class SearchApiHtmxRow extends RowPluginBase implements ContainerFactoryPluginInterface {

  protected IndexInterface $index;

  /**
   * {@inheritDoc}
   */
  public function __construct(
    array $configuration, $plugin_id, $plugin_definition,
    private readonly EntityTypeManagerInterface $entityTypeManager,
    private readonly ComponentPluginManager $sdcManager,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritDoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration, $plugin_id, $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('plugin.manager.sdc'),
    );
  }

  /**
   * {@inheritDoc}
   */
  public function init(ViewExecutable $view, DisplayPluginBase $display, array &$options = NULL) {
    parent::init($view, $display, $options);
    $base_table = $view->storage->get('base_table');
    $this->index = SearchApiQuery::getIndexFromTable($base_table, $this->entityTypeManager);
  }

  /**
   * {@inheritdoc}
   */
  protected function defineOptions() {
    $options = parent::defineOptions();
    $options['single_directory_components'] = ['default' => []];
    $options['extra_htmx_attributes'] = ['default' => [
      'data-hx-trigger' => 'revealed',
      'data-hx-swap' => 'outerHTML',
    ]];
    $options['extra_libraries'] = ['default' => ''];
    $options['initial_load_count'] = ['default' => 0];

    return $options;
  }

  /**
   * {@inheritDoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state): void {
    parent::buildOptionsForm($form, $form_state);

    foreach ($this->index->getDatasources() as $datasource_id => $datasource) {
      $view_modes = $datasource->getViewModes();

      $title = $this->t('View mode for datasource %datasource', [
        '%datasource' => $datasource->label(),
      ]);

      $form['view_modes'][$datasource_id] = ['#type' => 'container'];
      $form['view_modes'][$datasource_id]['view_mode'] = [
        '#type' => 'select',
        '#options' => $view_modes,
        '#title' => $title,
        '#default_value' => key($view_modes),
      ];

      if (isset($this->options['view_modes'][$datasource_id]['view_mode'])) {
        $form['view_modes'][$datasource_id]['view_mode']['#default_value'] = $this->options['view_modes'][$datasource_id]['view_mode'];
      }
    }

    $form['single_directory_components'] = [
      '#type' => 'select',
      '#title' => $this->t('Single directory components'),
      '#description' => $this->t("Select the SDC's that will be loaded in, so we can statically preload the relevant assets."),
      '#options' => array_map(static function ($sdc) {
        return "{$sdc['name']} ({$sdc['id']})";
      }, $this->sdcManager->getDefinitions()),
      '#default_value' => $this->options['single_directory_components'] ?? [],
      '#multiple' => TRUE,
    ];

    $form['extra_libraries'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Extra libraries'),
      '#description' => $this->t("Enter one asset library per line."),
      '#default_value' => $this->options['extra_libraries'] ?? '',
    ];

    $extra_htmx_attributes = '';
    foreach ($this->options['extra_htmx_attributes'] ?? [] as $attribute => $value) {
      if (is_array($value)) continue;
      $extra_htmx_attributes .= $attribute . ':' . $value . "\n";
    }
    $form['extra_htmx_attributes'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Extra HTMX attributes'),
      '#description' => $this->t('Enter one attribute per line, and split attribute from value like so: "attribute:value".
See <a href=":url" target="_blank">HTMX.org</a> for valid attributes.<br>Tip: for scrollable divs using overflows, use "trigger:intersect once" rather than "trigger:revealed"', [
        ':url' => 'https://htmx.org/reference/',
      ]),
      '#placeholder' => "data-hx-trigger:revealed\r\ndata-hx-swap:outerHTML",
      '#default_value' => $extra_htmx_attributes,
    ];

    $form['initial_load_count'] = [
      '#type' => 'number',
      '#title' => $this->t('Initial load count'),
      '#description' => $this->t('The number of items to load on page load, in case you used any other trigger than "load" in the htmx attributes.'),
      '#default_value' => $this->options['initial_load_count'],
    ];
  }

  /**
   * {@inheritDoc}
   */
  public function validateOptionsForm(&$form, FormStateInterface $form_state) {
    $values =& $form_state->getValue('row_options');
    $extra_htmx_attributes_exploded = explode("\n", $values['extra_htmx_attributes']);
    $extra_htmx_attributes_exploded = array_filter(array_map('trim', $extra_htmx_attributes_exploded));
    $extra_htmx_attributes = [];
    $valid_attributes = HtmxExtrasHelper::availableAttributes();
    $used_attributes = [];
    foreach ($extra_htmx_attributes_exploded as $attr) {
      [$attribute, $value] = explode(':', $attr, 2);
      $attribute = strtolower($attribute);
      if (!in_array($attribute, $valid_attributes)) {
        $form_state->setErrorByName('extra_htmx_attributes', $this->t("Invalid HTMX-attribute: '@attribute'.", ['@attribute' => $attribute]));
        continue;
      }
      if (in_array($attribute, $used_attributes)) {
        $form_state->setErrorByName('extra_htmx_attributes', $this->t("Duplicate attribute: '@attribute'.", ['@attribute' => $attribute]));
        continue;
      }
      $extra_htmx_attributes[$attribute] = $value;
      $used_attributes[] = $attribute;
    }
    $values['extra_htmx_attributes'] = $extra_htmx_attributes;
    parent::validateOptionsForm($form, $form_state);
  }

  /**
   * {@inheritDoc}
   */
  public function render($row): array {
    [$datasource_id] = Utility::splitCombinedId($row->_item->getId());
    $view_mode = $this->options['view_modes'][$datasource_id]['view_mode'] ?? 'default';
    /** @var \Drupal\Core\Entity\EntityInterface $entity */
    if (!$entity = $row->_entity) {
      return [];
    }

    $partial = HtmxEntityPartial::fromEntity($entity)
      ->setViewMode($view_mode)
      ->setAttachment('library', $this->getLibraries());

    // Load a fixed number of items immediately on page load.
    $extra_htmx_attributes = $this->options['extra_htmx_attributes'];
    if ($row->index < ($this->options['initial_load_count'] ?? 0)) {
      $extra_htmx_attributes['data-hx-trigger'] = 'load';
    }
    foreach ($extra_htmx_attributes as $attribute => $value) {
      $partial->setHtmxParameter($attribute, $value);
    }

    $build = $partial->render();

    // Add the excerpt to the render array to allow adding it to view modes.
    if (isset($row->search_api_excerpt)) {
      $build['#search_api_excerpt'] = $row->search_api_excerpt;
    }

    return $build;
  }

  private function getLibraries(): array {
    $sdc_libraries = array_map(static function ($sdc) {
      return 'core/components.' . str_replace(':', '--', $sdc);
    }, $this->options['single_directory_components'] ?? []);

    $extra_libraries = array_map(static function ($library) {
      return str_replace("\r", '', $library);
    }, explode("\n", $this->options['extra_libraries'] ?? ''));

    return array_values(array_filter(array_merge($sdc_libraries, $extra_libraries)));
  }

}
