<?php

namespace Drupal\htmx_extras\Render;

use Drupal\Core\Cache\CacheableDependencyInterface;
use Drupal\Core\Cache\RefinableCacheableDependencyInterface;
use Drupal\Core\Cache\RefinableCacheableDependencyTrait;
use Drupal\htmx_extras\HtmxExtrasHelper;

final class HtmxRoutePartial implements RefinableCacheableDependencyInterface {

  use RefinableCacheableDependencyTrait;

  public function __construct(
    private readonly string $routeName,
    private readonly array $routeParameters = [],
    private readonly array $routeOptions = [],
    private array $extraHtmxAttributes = [
      'trigger' => 'revealed',
      'swap' => 'outerHTML',
    ],
    private array $attachments = [],
    ?CacheableDependencyInterface $cache = NULL,
  ) {
    if ($cache) {
      $this->setCacheability($cache);
    }
  }

  public static function fromRoute(
    string $route_name,
    array $route_parameters = [],
    array $route_options = [],
  ): self {
    return new self(
      routeName: $route_name,
      routeParameters: $route_parameters,
      routeOptions: $route_options,
    );
  }

  /**
   * Set a HTMX parameter (excluding the data-hx or hx- prefixes).
   *
   * @see \Drupal\htmx_extras\HtmxExtrasHelper::availableAttributes()
   *   For the available attributes/directives.
   *
   * @throws \BadMethodCallException
   */
  public function setHtmxParameter(string $key, string $value): self {
    $key = str_replace(['data-hx-', 'hx-'], '', $key);
    if (!in_array($key, HtmxExtrasHelper::availableAttributes(with_prefix: FALSE))) {
      throw new \BadMethodCallException('This HTMX directive does not exist.');
    }

    $this->extraHtmxAttributes[$key] = $value;
    return $this;
  }

  /**
   * Set a route/url option, note that the language is usually already set.
   */
  public function setOption(string $key, string $value): self {
    $this->options[$key] = $value;
    return $this;
  }

  /**
   * Add an attachment to a given subset.
   */
  public function addAttachment(string $key, mixed $value): self {
    $this->attachments[$key][] = $value;
    return $this;
  }

  /**
   * Override an entire attachment key.
   */
  public function setAttachment(string $key, mixed $value): self {
    $this->attachments[$key] = $value;
    return $this;
  }

  /**
   * Returns the final template render array.
   */
  public function render(): array {
    $extra_htmx_attributes = [];
    foreach ($this->extraHtmxAttributes as $key => $value) {
      if (str_starts_with($key, 'data-hx-')) {
        $extra_htmx_attributes[$key] = $value;
      }
      elseif (str_starts_with($key, 'hx-')) {
        $extra_htmx_attributes["data-$key"] = $value;
      }
      else {
        $extra_htmx_attributes["data-hx-$key"] = $value;
      }
    }
    return [
      '#theme' => 'htmx_route_partial',
      '#route_name' => $this->routeName,
      '#route_parameters' => $this->routeParameters,
      '#route_options' => $this->routeOptions,
      '#extra_htmx_attributes' => $this->extraHtmxAttributes,
      '#cache' => [
        'tags' => $this->getCacheTags(),
        'contexts' => $this->getCacheContexts(),
        'max-age' => $this->getCacheMaxAge(),
      ],
      '#attached' => $this->attachments,
    ];
  }

}
