<?php

namespace Drupal\http_client_log\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;

/**
 * Configure http_client_log settings for this site.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'http_client_log_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['http_client_log.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('http_client_log.settings');

    $key = 'log_retention_limit';
    $row_limits = [1000, 10000, 100000, 200000, 300000, 500000, 1000000];
    $form[$key] = [
      '#type' => 'select',
      '#title' => $this->t('Http client logs to keep'),
      '#default_value' => $config->get($key),
      '#options' => [0 => $this->t('All')] + array_combine($row_limits, $row_limits),
      '#description' => $this->t('The maximum number of messages to keep in the database log. Requires a <a href=":cron">cron maintenance task</a>.', [
        ':cron' => Url::fromRoute('system.status')->toString(),
      ]),
    ];

    $key = 'response_content_type_filter';
    $form[$key] = [
      '#type' => 'textarea',
      '#title' => $this->t('Filter recording logs by response content type'),
      '#description' => $this->t('Only record logs when the response has a <a href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Content-Type#syntax">HTTP Content-Type header</a> containing at least one of these entries. Each entry should be on its own line.'),
      '#default_value' => implode("\n", array_filter($config->get($key) ?? [])),
    ];

    $key = 'only_log_when_response';
    $form[$key] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Only record logs when there is a response'),
      '#default_value' => $config->get($key),
      '#states' => [
        'invisible' => [
          ':input[name="response_content_type_filter"]' => ['filled' => TRUE],
        ],
      ],
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->config('http_client_log.settings')
      ->set('log_retention_limit', $form_state->getValue('log_retention_limit'))
      ->set('only_log_when_response', $form_state->getValue('only_log_when_response'))
      ->set('response_content_type_filter', array_filter(array_map(
        'trim',
        explode("\n", $form_state->getValue('response_content_type_filter'))
      )))
      ->save();
    parent::submitForm($form, $form_state);
  }

}
