<?php

namespace Drupal\http_client_log\Logger;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\DependencyInjection\DependencySerializationTrait;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\RfcLoggerTrait;
use GuzzleHttp\Psr7\Response;
use Psr\Log\LoggerInterface;

/**
 * Defines Logger class.
 */
class Logger implements LoggerInterface {

  use RfcLoggerTrait;
  use DependencySerializationTrait;

  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected ConfigFactoryInterface $configFactory,
    protected TimeInterface $time,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function log($level, $message, array $context = []) :void {
    /** @var \GuzzleHttp\Psr7\Request $request */
    $request = $context['request'];
    /** @var \GuzzleHttp\Psr7\Response $response */
    $response = $context['response'];
    $config = $this->configFactory->get('http_client_log.settings');

    if (!$response && $config->get('only_log_when_response')) {
      return;
    }

    // If there is a content type filter, only log if the response has a content
    // type containing one of the strings in the filter.
    $content_type_filter = $config->get('response_content_type_filter');
    if (
      !empty($content_type_filter)
      && $response instanceof Response
    ) {
      $content_type = $response->getHeader('Content-Type')[0] ?? '';

      $match = FALSE;
      foreach ($content_type_filter as $filter_item) {
        // The Content-Type header is not case sensitive. See
        // https://www.w3.org/Protocols/rfc1341/4_Content-Type.html.
        if (stripos($content_type, $filter_item) !== FALSE) {
          $match = TRUE;
          break;
        }
      }
      if (!$match) {
        return;
      }
    }

    $request_headers = [];
    foreach ($request->getHeaders() as $req_k => $req_v) {
      $request_headers[] = $req_k . ': ' . implode('|', $req_v);
    }

    // See discussion here:
    // https://github.com/guzzle/guzzle/pull/1262#issuecomment-149080749
    $request_body_stream = $request->getBody();
    $requestPos = $request_body_stream->tell();

    // Rewind if seekable, then read contents.
    if ($request_body_stream->isSeekable()) {
      $request_body_stream->rewind();
    }
    $request_payload = $request_body_stream->getContents();

    $field_values = [
      'type' => 'http_client_log',
      'changed' => $this->time->getCurrentTime(),
      'request_http_method' => $request->getMethod(),
      'request_url' => $request->getUri(),
      'request_headers' => implode("\r\n", $request_headers),
      'request_payload' => $request_payload,
      'context' => $context,
    ];

    if ($request_body_stream->isSeekable() && $requestPos != $request_body_stream->tell()) {
      $request_body_stream->rewind();
    }

    if ($response instanceof Response) {
      $response_body_stream = $response->getBody();
      $responsePos = $response_body_stream->tell();

      $field_values += [
        'response_status_code' => $response->getStatusCode(),
        'response_reason_phrase' => $response->getReasonPhrase(),
      ];

      $response_headers = [];
      foreach ($response->getHeaders() as $res_k => $res_v) {
        $response_headers[] = $res_k . ': ' . implode('|', $res_v);
      }
      $field_values['response_headers'] = implode("\r\n", $response_headers);

      $field_values['response_body'] = $response_body_stream->getContents();
      if ($response_body_stream->isSeekable() && $responsePos != $response_body_stream->tell()) {
        $response_body_stream->rewind();
      }
    }

    $this->entityTypeManager->getStorage('http_client_log')
      ->create($field_values)
      ->save();
  }

}
