<?php

declare(strict_types=1);

namespace Drupal\icon_media_pack\Plugin\IconExtractor;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Theme\Icon\Attribute\IconExtractor;
use Drupal\Core\Theme\Icon\IconDefinition;
use Drupal\Core\Theme\Icon\IconDefinitionInterface;
use Drupal\Core\Theme\Icon\IconExtractorBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the icon_extractor.
 */
#[IconExtractor(
  id: 'media_entity',
  label: new TranslatableMarkup('Media Entity'),
  description: new TranslatableMarkup('Extract icons from media entities.'),
)]
class MediaEntityExtractor extends IconExtractorBase implements ContainerFactoryPluginInterface {

  /**
   * Constructs a MediaEntityExtractor object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin ID for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer service.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected RendererInterface $renderer,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): self {
    return new self(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('renderer')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function discoverIcons(): array {
    $media_bundle = $this->configuration['config']['media_bundle'] ?? NULL;

    if (!$media_bundle) {
      return [];
    }

    $media_storage = $this->entityTypeManager->getStorage('media');
    $view_builder = $this->entityTypeManager->getViewBuilder('media');

    // Load all media entities of the specified bundle.
    $query = $media_storage->getQuery()
      ->condition('bundle', $media_bundle)
      ->condition('status', 1)
      ->accessCheck(FALSE)
      ->sort('name', 'ASC');

    $media_ids = $query->execute();

    if (empty($media_ids)) {
      return [];
    }

    $media_entities = $media_storage->loadMultiple($media_ids);
    $icons = [];

    foreach ($media_entities as $media) {
      $icon_id = $this->sanitizeIconId($media->label());
      $id = IconDefinition::createIconId($this->configuration['id'], $icon_id);

      $source = $this->getSourceFile($media);

      $icons[$id] = [
        'source' => (string) $source,
        'group' => $media_bundle,
        'media_id' => $media->id(),
        'media_uuid' => $media->uuid(),
      ];
    }

    return $icons;
  }

  /**
   * {@inheritdoc}
   */
  public function loadIcon(array $icon_data): ?IconDefinitionInterface {
    if (!isset($icon_data['media_id'])) {
      return NULL;
    }

    $media_storage = $this->entityTypeManager->getStorage('media');
    $media = $media_storage->load($icon_data['media_id']);

    if (!$media) {
      return NULL;
    }

    // Get the media source field.
    $source_field = $media->getSource()->getSourceFieldDefinition($media->bundle->entity);
    $field_name = $source_field->getName();

    if (!$media->hasField($field_name)) {
      return NULL;
    }

    $source = NULL;
    $content = NULL;

    // Handle different field types.
    $field_type = $source_field->getType();

    if ($field_type === 'image') {
      // For image fields, get the file URL.
      $field_item = $media->get($field_name)->first();
      if ($field_item && $field_item->entity) {
        $file = $field_item->entity;
        $source = $file->createFileUrl();
      }
    }
    elseif ($field_type === 'ui_icon') {
      // For icon fields, the value is the icon reference.
      $content = $media->get($field_name)->value;
    }
    else {
      // Fallback: try to get a simple value.
      $content = $media->get($field_name)->value;
    }
    $view_builder = $this->entityTypeManager->getViewBuilder('media');
    $render_array = $view_builder->view($media, 'icon_media_pack');
    $content = $this->renderer->renderInIsolation($render_array);

    return $this->createIcon(
      $icon_data['icon_id'],
      $source,
      $icon_data['group'] ?? NULL,
      $content ? ['content' => $render_array] : NULL,
    );
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $media_types = $this->entityTypeManager->getStorage('media_type')->loadMultiple();
    $options = [];
    foreach ($media_types as $media_type) {
      $options[$media_type->id()] = $media_type->label();
    }

    $form['media_bundle'] = [
      '#type' => 'select',
      '#title' => t('Media bundle'),
      '#description' => t('Select the media bundle to use as icon source.'),
      '#options' => $options,
      '#default_value' => $this->configuration['media_bundle'] ?? '',
      '#required' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    parent::submitConfigurationForm($form, $form_state);

    $this->configuration['media_bundle'] = $form_state->getValue('media_bundle');
  }

  /**
   * Sanitize icon ID to be URL-safe.
   *
   * @param string $label
   *   The media label.
   *
   * @return string
   *   The sanitized icon ID.
   */
  protected function sanitizeIconId(string $label): string {
    // Convert to lowercase and replace spaces/special chars with hyphens.
    $id = strtolower($label);
    $id = preg_replace('/[^a-z0-9]+/', '-', $id);
    $id = trim($id, '-');

    return $id;
  }

  /**
   * Get the source file URL from a media entity.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The media entity.
   *
   * @return string|null
   *   The file URL or NULL if not found.
   */
  protected function getSourceFile(EntityInterface $entity): ?string {
    // Get the media source field.
    $source_field = $entity->getSource()->getSourceFieldDefinition($entity->bundle->entity);
    $field_name = $source_field->getName();

    if (!$entity->hasField($field_name)) {
      return NULL;
    }

    // Handle different field types.
    $field_type = $source_field->getType();

    if ($field_type === 'image') {
      // For image fields, get the file URL.
      $field_item = $entity->get($field_name)->first();
      if ($field_item && $field_item->entity) {
        $file = $field_item->entity;
        return $file->createFileUrl();
      }
    }

    return NULL;

  }

}
