<?php

declare(strict_types=1);

namespace Drupal\idle_reauthenticate\Controller;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\CloseModalDialogCommand;
use Drupal\Core\Ajax\OpenModalDialogCommand;
use Drupal\Core\Ajax\RedirectCommand;
use Drupal\Core\Ajax\SettingsCommand;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Password\PasswordInterface;
use Drupal\Core\Session\SessionManagerInterface;
use Drupal\Core\TempStore\PrivateTempStore;
use Drupal\Core\Url;
use Drupal\idle_reauthenticate\Form\Authenticate;
use Drupal\masquerade\Masquerade;
use Drupal\user\Entity\User;
use Drupal\user\UserInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Returns responses for Reauthenticate on idle browser routes.
 */
final class PingBack extends ControllerBase {

  public const string STATE_BLOCKED = 'blocked';
  public const string STATE_UNBLOCKED = 'unblocked';
  public const string STATE_UNKNOWN = 'unknown';

  /**
   * The ajax response.
   *
   * @var \Drupal\Core\Ajax\AjaxResponse
   */
  protected AjaxResponse $response;

  /**
   * The current user.
   *
   * @var \Drupal\user\UserInterface
   */
  protected UserInterface $sessionUser;

  /**
   * The current user, or the user who's masquerading as the current user.
   *
   * @var \Drupal\user\UserInterface
   */
  protected UserInterface $user;

  /**
   * The controller constructor.
   */
  public function __construct(
    protected readonly Request $request,
    protected readonly PrivateTempStore $store,
    protected readonly PasswordInterface $passwordChecker,
    protected readonly ImmutableConfig $config,
    protected readonly SessionManagerInterface $session,
    protected readonly ?Masquerade $masquerade,
  ) {
    $this->sessionUser = User::load($this->currentUser()->id());
    if ($this->isMasquerading()) {
      // @phpstan-ignore-next-line
      $this->user = User::load($this->session->getMetadataBag()->getMasquerade());
    }
    else {
      $this->user = $this->sessionUser;
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    $masquerade = $container->get('module_handler')->moduleExists('masquerade') ?
      $container->get('masquerade') :
      NULL;
    return new self(
      $container->get('request_stack')->getCurrentRequest(),
      $container->get('tempstore.private')->get('idle_reauthenticate'),
      $container->get('password'),
      $container->get('config.factory')->get('idle_reauthenticate.settings'),
      $container->get('session_manager'),
      $masquerade,
    );
  }

  /**
   * Determines if the current user is being masqueraded.
   *
   * @return bool
   *   TRUE, if the masquerade module is installed and if the current user is
   *   being masqueraded.
   */
  protected function isMasquerading(): bool {
    return $this->masquerade !== NULL && $this->masquerade->isMasquerading();
  }

  /**
   * Get the session related key for the given key.
   *
   * @param string $key
   *   The key.
   *
   * @return string
   *   The session related key.
   */
  protected function getKey(string $key): string {
    return $this->session->getId() . ':' . $key;
  }

  /**
   * Access controller for the ping back.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   Always allow access because we have to allow requests from e.g. inactive
   *   browser tabs when the focused tab logged out the current user. In such
   *   cases, the ping should be allowed but then also perform a redirect to the
   *   login page.
   */
  public function access(): AccessResultInterface {
    return AccessResult::allowed();
  }

  /**
   * Builds the response.
   */
  public function __invoke(): AjaxResponse {
    $this->response = new AjaxResponse();
    $payload = $this->request->getPayload();
    $currentPath = $payload->get('currentPath', Url::fromRoute('<front>')->toString());
    if (!$this->currentUser()->isAuthenticated()) {
      $this->reLogin($currentPath);
      return $this->response;
    }
    if ($payload->has('form_id') && $payload->get('form_id') === 'idle_reauthenticate_authenticate') {
      if ($payload->get('_triggering_element_name') === 'logout' || $this->sessionUser->isBlocked() || $this->user->isBlocked()) {
        $this->reLogin($currentPath);
      }
      else {
        if ($this->passwordChecker->check($payload->get('password'), $this->user->getPassword())) {
          $this->unblockSession();
        }
      }
      return $this->response;
    }

    $lastSeen = $payload->get('lastSeen', 0);
    $currentTime = $payload->get('currentTime', FALSE);
    $currentState = $payload->get('currentState', FALSE);

    $dbState = $this->store->get($this->getKey('state')) ?? self::STATE_UNKNOWN;
    $dbLastSeen = $this->store->get($this->getKey('lastSeen')) ?? 0;
    if ($currentTime === FALSE) {
      $this->blockSession('');
    }
    elseif ($currentState === FALSE) {
      $this->store->set($this->getKey('lastSeen'), $currentTime);
      if ($dbState === self::STATE_BLOCKED) {
        $this->blockSession($currentPath);
      }
      else {
        $this->unblockSession(FALSE);
      }
    }
    else {
      if ($dbLastSeen > $lastSeen) {
        $lastSeen = $dbLastSeen;
      }
      else {
        $this->store->set($this->getKey('lastSeen'), $lastSeen);
      }
      if ($currentState === self::STATE_UNBLOCKED
        && ($currentTime - $lastSeen) > $this->config->get('idle_period') * 1000
      ) {
        $this->blockSession($currentPath);
      }
      elseif ($currentState === self::STATE_BLOCKED && $dbState === self::STATE_UNBLOCKED) {
        $this->unblockSession();
      }
    }
    return $this->response;
  }

  /**
   * Set the session blocked.
   *
   * @param string $destination
   *   The destination path where to redirect the user after successful login.
   */
  private function blockSession(string $destination): void {
    if (($this->store->get($this->getKey('state')) ?? self::STATE_UNKNOWN) !== self::STATE_BLOCKED) {
      $this->store->set($this->getKey('state'), self::STATE_BLOCKED);
    }
    $this->response->addCommand(new SettingsCommand(['idleReauthenticate' => ['currentState' => self::STATE_BLOCKED]], TRUE));
    $form = $this->formBuilder()->getForm(Authenticate::class);
    $form['current_path'] = ['#type' => 'hidden', '#value' => $destination, '#name' => 'currentPath'];
    $displayName = $this->isMasquerading() ?
      $this->t('%name masquerading as %currentName', [
        '%name' => $this->user->getDisplayName(),
        '%currentName' => $this->sessionUser->getDisplayName(),
      ]) :
      $this->sessionUser->getDisplayName();
    if ($this->isMasquerading()) {
      $form['actions']['logout']['#value'] = $this->t('Not %name?', ['%name' => $displayName]);
    }
    $this->response->addCommand(new OpenModalDialogCommand(
      $this->t('Re-authenticate %name', ['%name' => $displayName]),
      $form,
      [
        'classes' => [
          'ui-dialog' => 'idle-reauthenticate',
        ],
        'closeOnEscape' => FALSE,
      ]));
  }

  /**
   * Set the session unblocked.
   */
  private function unblockSession(bool $closeDialog = TRUE): void {
    if (($this->store->get($this->getKey('state')) ?? self::STATE_UNKNOWN) !== self::STATE_UNBLOCKED) {
      $this->store->set($this->getKey('state'), self::STATE_UNBLOCKED);
    }
    $this->response->addCommand(new SettingsCommand(['idleReauthenticate' => ['currentState' => self::STATE_UNBLOCKED]], TRUE));
    if ($closeDialog) {
      $this->response->addCommand(new CloseModalDialogCommand());
    }
  }

  /**
   * Logout the current user and redirect to the login page.
   *
   * @param string $destination
   *   The destination path where to redirect the user after successful login.
   */
  private function reLogin(string $destination): void {
    if ($this->currentUser()->isAuthenticated()) {
      user_logout();
      $this->store->delete($this->getKey('state'));
    }
    $this->response->addCommand(new RedirectCommand(Url::fromRoute('user.login', [], ['query' => ['destination' => $destination]])->toString()));
  }

}
