<?php

declare(strict_types=1);

namespace Drupal\Tests\ief_table_view\Kernel;

use Drupal\Core\Entity\EntityStorageException;
use Drupal\Core\Form\FormState;
use Drupal\entity_test\Entity\EntityTest;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\Tests\field\Kernel\FieldKernelTestBase;
use Drupal\views\Entity\View;

/**
 * Kernel tests for InlineEntityFormComplexTableView.
 *
 * @coversDefaultClass \Drupal\ief_table_view\Plugin\Field\FieldWidget\InlineEntityFormComplexTableView
 * @group ief_table_view
 */
class InlineEntityFormComplexTableViewTest extends FieldKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'entity_test',
    'ief_table_view',
    'inline_entity_form',
    'views',
  ];

  /**
   * The field storage.
   *
   * @var \Drupal\field\FieldStorageConfigInterface
   */
  protected $fieldStorage;

  /**
   * The field configuration.
   *
   * @var \Drupal\field\FieldConfigInterface
   */
  protected $field;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    // Disable strict config schema checking for Views.
    $this->strictConfigSchema = FALSE;
    parent::setUp();
    $this->installEntitySchema('entity_test');
    $this->installEntitySchema('user');
    $this->installConfig(['field']);

    // Create an entity reference field.
    $this->fieldStorage = FieldStorageConfig::create([
      'field_name' => 'test_reference',
      'entity_type' => 'entity_test',
      'type' => 'entity_reference',
      'settings' => [
        'target_type' => 'entity_test',
      ],
    ]);
    try {
      $this->fieldStorage->save();
    }
    catch (EntityStorageException $e) {
      $this->fail('Failed to create field storage: ' . $e->getMessage());
    }

    $this->field = FieldConfig::create([
      'field_name' => 'test_reference',
      'entity_type' => 'entity_test',
      'bundle' => 'entity_test',
    ]);
    try {
      $this->field->save();
    }
    catch (EntityStorageException $e) {
      $this->fail('Failed to create field: ' . $e->getMessage());
    }
  }

  /**
   * Gets the widget plugin instance with proper configuration.
   */
  protected function getWidget(): object {
    try {
      return $this->container->get('plugin.manager.field.widget')
        ->createInstance('ief_table_view_complex', [
          'field_definition' => $this->field,
          'settings' => [],
          'third_party_settings' => [],
        ]);
    }
    catch (\Exception $e) {
      $this->fail('Failed to create widget: ' . $e->getMessage());
    }
  }

  /**
   * Tests default settings.
   *
   * @covers ::defaultSettings
   */
  public function testDefaultSettings(): void {
    $widget = $this->getWidget();
    $settings = $widget->defaultSettings();

    $this->assertArrayHasKey('view_display', $settings);
    $this->assertNull($settings['view_display']);
  }

  /**
   * Tests settings form contains view_display selector.
   *
   * @covers ::settingsForm
   */
  public function testSettingsForm(): void {
    $widget = $this->getWidget();
    $form_state = new FormState();
    $form = [];

    $settings_form = $widget->settingsForm($form, $form_state);

    $this->assertArrayHasKey('view_display', $settings_form);
    $this->assertEquals('select', $settings_form['view_display']['#type']);
    $this->assertEquals('Entity table view/display', $settings_form['view_display']['#title']);
    $this->assertArrayHasKey('#required', $settings_form['view_display']);
  }

  /**
   * Tests it settings summary when the view display is not set.
   *
   * @covers ::settingsSummary
   */
  public function testSettingsSummaryNotConfigured(): void {
    $widget = $this->getWidget();
    try {
      $summary = $widget->settingsSummary();
    }
    catch (\Exception $e) {
      $this->fail('Failed to get settings summary: ' . $e->getMessage());
    }

    $found = FALSE;
    foreach ($summary as $item) {
      if (str_contains((string) $item, 'Entity table view:')) {
        $found = TRUE;
        break;
      }
    }
    $this->assertTrue($found, 'Summary should contain "Entity table view:"');
  }

  /**
   * Tests it settings summary when the view display is set.
   *
   * @covers ::settingsSummary
   */
  public function testSettingsSummaryConfigured(): void {
    $this->createTestView();
    $widget = $this->getWidget();
    $widget->setSetting('view_display', 'ief_test_view:ief_table');

    try {
      $summary = $widget->settingsSummary();
    }
    catch (\Exception $e) {
      $this->fail('Failed to get settings summary: ' . $e->getMessage());
    }

    $this->assertNotEmpty($summary);
    $found = FALSE;
    foreach ($summary as $item) {
      if (str_contains((string) $item, 'IEF Test View')) {
        $found = TRUE;
        break;
      }
    }
    $this->assertTrue($found, 'Summary should contain the view label');
  }

  /**
   * Tests calculateDependencies includes the view.
   *
   * @covers ::calculateDependencies
   */
  public function testCalculateDependencies(): void {
    $this->createTestView();
    $widget = $this->getWidget();
    $widget->setSetting('view_display', 'ief_test_view:ief_table');

    $dependencies = $widget->calculateDependencies();

    $this->assertArrayHasKey('config', $dependencies);
    $this->assertContains('views.view.ief_test_view', $dependencies['config']);
  }

  /**
   * Tests that widget can be instantiated.
   *
   * @covers ::__construct
   */
  public function testWidgetInstantiation(): void {
    $widget = $this->getWidget();
    $this->assertNotNull($widget);
  }

  /**
   * Tests formElement adds process callback.
   *
   * @covers ::formElement
   */
  public function testFormElement(): void {
    $widget = $this->getWidget();

    // Create a real entity to avoid complex mocking.
    $entity = EntityTest::create([]);
    $items = $entity->get('test_reference');

    $form = [];
    $form_state = new FormState();

    $element = $widget->formElement($items, 0, ['#field_parents' => [], '#required' => FALSE], $form, $form_state);

    $this->assertArrayHasKey('#process', $element);
    $this->assertContains([$widget, 'processForm'], $element['#process']);
  }

  /**
   * Creates a test view with IEF table display.
   */
  protected function createTestView(): void {
    try {
      View::create([
        'id' => 'ief_test_view',
        'label' => 'IEF Test View',
        'base_table' => 'entity_test',
        'display' => [
          'default' => [
            'display_plugin' => 'default',
            'id' => 'default',
            'display_title' => 'Default',
            'position' => 1,
            'display_options' => [
              'style' => [
                'type' => 'ief_table',
              ],
            ],
          ],
          'ief_table' => [
            'display_plugin' => 'ief_table',
            'id' => 'ief_table',
            'display_title' => 'IEF Table',
            'position' => 2,
            'display_options' => [
              'style' => [
                'type' => 'ief_table',
              ],
            ],
          ],
        ],
      ])->save();
    }
    catch (EntityStorageException $e) {
      $this->fail('Failed to create test view: ' . $e->getMessage());
    }
  }

}
