<?php

declare(strict_types=1);

namespace Drupal\Tests\ief_table_view\Kernel;

use Drupal\Core\Entity\EntityStorageException;
use Drupal\entity_test\Entity\EntityTest;
use Drupal\views\Entity\View;
use Drupal\views\ResultRow;
use Drupal\Tests\views\Kernel\ViewsKernelTestBase;

/**
 * Kernel tests for InlineEntityFormTableDisplay.
 *
 * @coversDefaultClass \Drupal\ief_table_view\Plugin\views\display\InlineEntityFormTableDisplay
 * @group ief_table_view
 */
class InlineEntityFormTableDisplayTest extends ViewsKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'entity_test',
    'ief_table_view',
    'inline_entity_form',
    'user',
    'views',
  ];

  /**
   * The view executable.
   *
   * @var \Drupal\views\ViewExecutable
   */
  protected $view;

  /**
   * {@inheritdoc}
   */
  protected function setUp($import_test_views = TRUE): void {
    // Disable strict config schema checking for Views.
    $this->strictConfigSchema = FALSE;
    parent::setUp($import_test_views);
    $this->installEntitySchema('entity_test');
    $this->installEntitySchema('user');

    // Create a test view with the IEF table display.
    $view = View::create([
      'id' => 'ief_test_view',
      'label' => 'IEF Test View',
      'base_table' => 'entity_test',
      'display' => [
        'default' => [
          'display_plugin' => 'default',
          'id' => 'default',
          'display_title' => 'Default',
          'position' => 1,
          'display_options' => [
            'style' => [
              'type' => 'ief_table',
            ],
          ],
        ],
        'ief_table' => [
          'display_plugin' => 'ief_table',
          'id' => 'ief_table',
          'display_title' => 'IEF Table',
          'position' => 2,
          'display_options' => [
            'style' => [
              'type' => 'ief_table',
            ],
          ],
        ],
      ],
    ]);
    try {
      $view->save();
    }
    catch (EntityStorageException $e) {
      $this->fail('Failed to create view: ' . $e->getMessage());
    }

    $this->view = $view->getExecutable();
  }

  /**
   * Tests the display type.
   *
   * @covers ::getType
   */
  public function testGetType(): void {
    $display = $this->view->displayHandlers->get('ief_table');
    $this->assertEquals('ief_table', $display->getType());
  }

  /**
   * Tests that AJAX is disabled.
   *
   * @covers ::usesExposed
   */
  public function testUsesExposed(): void {
    $display = $this->view->displayHandlers->get('ief_table');
    $this->assertFalse($display->usesExposed());
  }

  /**
   * Tests setting result entities adds missing entities.
   *
   * @covers ::setResultEntities
   */
  public function testSetResultEntitiesAddsMissing(): void {
    $display = $this->view->displayHandlers->get('ief_table');

    // Create test entities.
    $entity1 = EntityTest::create(['name' => 'Test 1']);
    try {
      $entity1->save();
    }
    catch (EntityStorageException $e) {
      // Log error on failure.
      $this->fail('Failed to create entity: ' . $e->getMessage());
    }

    $entity2 = EntityTest::create(['name' => 'Test 2']);
    // Don't save entity2 - it's unsaved.
    // Set up the initial view result with only entity1.
    $row = new ResultRow();
    $row->_entity = $entity1;
    $row->_relationship_entities = [];
    $row->index = 0;
    $this->view->result = [$row];

    // Add both entities to the display.
    $display->setResultEntities([$entity1, $entity2]);

    // Verify both entities are in the result.
    $this->assertCount(2, $this->view->result);
    $this->assertEquals($entity1->uuid(), $this->view->result[0]->_entity->uuid());
    $this->assertEquals($entity2->uuid(), $this->view->result[1]->_entity->uuid());
  }

  /**
   * Tests setting result entities updates existing entities.
   *
   * @covers ::setResultEntities
   */
  public function testSetResultEntitiesUpdatesExisting(): void {
    $display = $this->view->displayHandlers->get('ief_table');

    $entity1 = EntityTest::create(['name' => 'Original Name']);
    try {
      $entity1->save();
    }
    catch (EntityStorageException $e) {
      // Log error on failure.
      $this->fail('Failed to create entity: ' . $e->getMessage());
    }

    // Set up the initial view result.
    $row = new ResultRow();
    $row->_entity = $entity1;
    $row->_relationship_entities = [];
    $row->index = 0;
    $this->view->result = [$row];

    // Update the entity.
    $entity1->set('name', 'Updated Name');

    // Set result entities with the updated entity.
    $display->setResultEntities([$entity1]);

    // Verify the entity in the result was updated.
    $this->assertCount(1, $this->view->result);
    $this->assertEquals('Updated Name', $this->view->result[0]->_entity->label());
  }

  /**
   * Tests that getInlineEntityFormTableColumns delegates to style plugin.
   *
   * @covers ::getInlineEntityFormTableColumns
   */
  public function testGetInlineEntityFormTableColumns(): void {
    $display = $this->view->displayHandlers->get('ief_table');
    $columns = $display->getInlineEntityFormTableColumns();

    $this->assertIsArray($columns);
  }

}
