<?php

declare(strict_types=1);

namespace Drupal\Tests\ief_table_view\Kernel;

use Drupal\Core\Entity\EntityStorageException;
use Drupal\views\Entity\View;
use Drupal\Tests\views\Kernel\ViewsKernelTestBase;

/**
 * Kernel tests for InlineEntityFormTableStyle.
 *
 * @coversDefaultClass \Drupal\ief_table_view\Plugin\views\style\InlineEntityFormTableStyle
 * @group ief_table_view
 */
class InlineEntityFormTableStyleTest extends ViewsKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'entity_test',
    'ief_table_view',
    'inline_entity_form',
    'user',
    'views',
  ];

  /**
   * The view executable.
   *
   * @var \Drupal\views\ViewExecutable
   */
  protected $view;

  /**
   * {@inheritdoc}
   */
  protected function setUp($import_test_views = TRUE): void {
    // Disable strict config schema check for Views.
    $this->strictConfigSchema = FALSE;
    parent::setUp($import_test_views);
    $this->installEntitySchema('entity_test');
    $this->installEntitySchema('user');

    // Create a test view.
    $view = View::create([
      'id' => 'ief_style_test_view',
      'label' => 'IEF Style Test View',
      'base_table' => 'entity_test',
      'display' => [
        'default' => [
          'display_plugin' => 'default',
          'id' => 'default',
          'display_title' => 'Default',
          'position' => 1,
          'display_options' => [
            'fields' => [
              'id' => [
                'id' => 'id',
                'table' => 'entity_test',
                'field' => 'id',
                'entity_type' => 'entity_test',
                'entity_field' => 'id',
                'label' => 'ID',
                'exclude' => FALSE,
              ],
              'name' => [
                'id' => 'name',
                'table' => 'entity_test',
                'field' => 'name',
                'entity_type' => 'entity_test',
                'entity_field' => 'name',
                'label' => 'Name',
                'exclude' => FALSE,
              ],
            ],
            'style' => [
              'type' => 'ief_table',
              'options' => [
                'columns' => [
                  'id' => 'id',
                  'name' => 'name',
                ],
                'info' => [
                  'id' => [
                    'sortable' => FALSE,
                    'default_sort_order' => 'asc',
                    'align' => '',
                    'separator' => '',
                    'empty_column' => FALSE,
                    'responsive' => '',
                  ],
                  'name' => [
                    'sortable' => FALSE,
                    'default_sort_order' => 'asc',
                    'align' => '',
                    'separator' => '',
                    'empty_column' => FALSE,
                    'responsive' => '',
                  ],
                ],
              ],
            ],
          ],
        ],
        'ief_table' => [
          'display_plugin' => 'ief_table',
          'id' => 'ief_table',
          'display_title' => 'IEF Table',
          'position' => 2,
          'display_options' => [
            'style' => [
              'type' => 'ief_table',
            ],
          ],
        ],
      ],
    ]);
    try {
      $view->save();
    }

    catch (EntityStorageException $e) {
      $this->fail('Failed to create view: ' . $e->getMessage());
    }

    $this->view = $view->getExecutable();
    $this->view->initDisplay();
    $this->view->initStyle();
    // Build the view to initialize field handlers.
    $this->view->build();
  }

  /**
   * Tests that grouping is disabled.
   *
   * @covers ::usesGrouping
   */
  public function testUsesGrouping(): void {
    $style = $this->view->style_plugin;
    $this->assertFalse($style->usesGrouping());
  }

  /**
   * Tests getInlineEntityFormTableColumns returns expected structure.
   *
   * @covers ::getInlineEntityFormTableColumns
   */
  public function testGetInlineEntityFormTableColumns(): void {
    $style = $this->view->style_plugin;
    $columns = $style->getInlineEntityFormTableColumns();

    $this->assertIsArray($columns);
    $this->assertArrayHasKey('id', $columns);
    $this->assertArrayHasKey('name', $columns);
  }

  /**
   * Tests column structure includes required keys.
   *
   * @covers ::getInlineEntityFormTableColumns
   */
  public function testColumnStructure(): void {
    $style = $this->view->style_plugin;
    $columns = $style->getInlineEntityFormTableColumns();

    foreach ($columns as $field => $column) {
      $this->assertArrayHasKey('label', $column);
      $this->assertArrayHasKey('type', $column);
      $this->assertArrayHasKey('callback', $column);
      $this->assertArrayHasKey('callback_arguments', $column);
      $this->assertArrayHasKey('weight', $column);

      $this->assertEquals('callback', $column['type']);
      $this->assertEquals('ief_table_view_inline_entity_form_entity_table_callback', $column['callback']);
      $this->assertArrayHasKey('style', $column['callback_arguments']);
      $this->assertArrayHasKey('field', $column['callback_arguments']);
      $this->assertEquals($field, $column['callback_arguments']['field']);
    }
  }

  /**
   * Tests that excluded fields are not included in columns.
   *
   * @covers ::getInlineEntityFormTableColumns
   */
  public function testExcludedFieldsNotIncluded(): void {
    // Create a view with an excluded field.
    $view = View::create([
      'id' => 'ief_excluded_test_view',
      'label' => 'IEF Excluded Test View',
      'base_table' => 'entity_test',
      'display' => [
        'default' => [
          'display_plugin' => 'default',
          'id' => 'default',
          'display_title' => 'Default',
          'position' => 1,
          'display_options' => [
            'fields' => [
              'id' => [
                'id' => 'id',
                'table' => 'entity_test',
                'field' => 'id',
                'entity_type' => 'entity_test',
                'entity_field' => 'id',
                'label' => 'ID',
                'exclude' => TRUE,
                'plugin_id' => 'field',
              ],
              'name' => [
                'id' => 'name',
                'table' => 'entity_test',
                'field' => 'name',
                'entity_type' => 'entity_test',
                'entity_field' => 'name',
                'label' => 'Name',
                'exclude' => FALSE,
                'plugin_id' => 'field',
              ],
            ],
            'style' => [
              'type' => 'ief_table',
              'options' => [
                'columns' => [
                  'id' => 'id',
                  'name' => 'name',
                ],
                'info' => [
                  'id' => ['separator' => ''],
                  'name' => ['separator' => ''],
                ],
              ],
            ],
          ],
        ],
      ],
    ]);
    try {
      $view->save();
    }
    catch (EntityStorageException $e) {
      $this->fail('Failed to create view: ' . $e->getMessage());
    }

    $executable = $view->getExecutable();
    $executable->initDisplay();
    $executable->initStyle();
    // Build the view to initialize field handlers.
    $executable->build();

    $columns = $executable->style_plugin->getInlineEntityFormTableColumns();

    $this->assertArrayNotHasKey('id', $columns, 'Excluded fields should not be in columns');
    $this->assertArrayHasKey('name', $columns);
  }

  /**
   * Tests that weights are assigned incrementally.
   *
   * @covers ::getInlineEntityFormTableColumns
   */
  public function testWeightsAreIncremental(): void {
    $style = $this->view->style_plugin;
    $columns = $style->getInlineEntityFormTableColumns();

    $expectedWeight = 0;
    foreach ($columns as $column) {
      $this->assertEquals($expectedWeight, $column['weight']);
      $expectedWeight++;
    }
  }

}
