<?php

declare(strict_types=1);

namespace Drupal\Tests\ief_table_view\Unit;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\ief_table_view\Render\EntityTableRenderer;
use Drupal\views\Plugin\views\style\StylePluginBase;
use Drupal\views\ResultRow;
use Drupal\views\ViewExecutable;
use Drupal\Tests\UnitTestCase;

/**
 * Unit tests for EntityTableRenderer.
 *
 * @coversDefaultClass \Drupal\ief_table_view\Render\EntityTableRenderer
 * @group ief_table_view
 */
class EntityTableRendererTest extends UnitTestCase {

  /**
   * The entity table renderer.
   *
   * @var \Drupal\ief_table_view\Render\EntityTableRenderer
   */
  protected $renderer;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->renderer = new EntityTableRenderer();
  }

  /**
   * Tests renderColumn with basic configuration.
   *
   * @covers ::renderColumn
   */
  public function testRenderColumnBasic(): void {
    $style = $this->createMockStylePlugin([
      'columns' => [
        'title' => 'title',
        'status' => 'status',
      ],
      'info' => [
        'title' => ['separator' => ' | '],
        'status' => ['separator' => ''],
      ],
    ]);

    $style->expects($this->any())
      ->method('getField')
      ->willReturnCallback(function ($index, $field) {
        return match ($field) {
          'title' => 'Test Title',
          'status' => 'Published',
          default => '',
        };
      });

    $result = $this->renderer->renderColumn($style, 0, 'title');

    $this->assertIsArray($result);
    $this->assertCount(1, $result);
    $this->assertArrayHasKey('#markup', $result[0]);
    $this->assertEquals('Test Title', $result[0]['#markup']);
  }

  /**
   * Tests renderColumn with multiple fields in the same column.
   *
   * @covers ::renderColumn
   */
  public function testRenderColumnMultipleFields(): void {
    $style = $this->createMockStylePlugin([
      'columns' => [
        'title' => 'title',
        'author' => 'title',
      ],
      'info' => [
        'title' => ['separator' => ' - '],
      ],
    ]);

    $style->expects($this->any())
      ->method('getField')
      ->willReturnCallback(function ($index, $field) {
        return match ($field) {
          'title' => 'Test Title',
          'author' => 'John Doe',
          default => '',
        };
      });

    $result = $this->renderer->renderColumn($style, 0, 'title');

    $this->assertIsArray($result);
    $this->assertCount(3, $result);
    $this->assertEquals('Test Title', $result[0]['#markup']);
    $this->assertEquals(' - ', $result[1]['#markup']);
    $this->assertEquals('John Doe', $result[2]['#markup']);
  }

  /**
   * Tests renderColumn with empty field output.
   *
   * @covers ::renderColumn
   */
  public function testRenderColumnEmptyField(): void {
    $style = $this->createMockStylePlugin([
      'columns' => ['title' => 'title'],
      'info' => ['title' => ['separator' => '']],
    ]);

    $style->expects($this->any())
      ->method('getField')
      ->willReturn('');

    $result = $this->renderer->renderColumn($style, 0, 'title');

    $this->assertIsArray($result);
    $this->assertEmpty($result);
  }

  /**
   * Tests renderEntityColumn finds entity by UUID.
   *
   * @covers ::renderEntityColumn
   */
  public function testRenderEntityColumn(): void {
    $entity = $this->createMockEntity('test-uuid-123');
    $style = $this->createMockStylePluginWithResult([
      'columns' => ['title' => 'title'],
      'info' => ['title' => ['separator' => '']],
    ], $entity);

    $style->expects($this->any())
      ->method('getField')
      ->willReturn('Entity Title');

    $result = $this->renderer->renderEntityColumn($style, $entity, 'title');

    $this->assertIsArray($result);
    $this->assertCount(1, $result);
    $this->assertEquals('Entity Title', $result[0]['#markup']);
  }

  /**
   * Tests renderEntityColumn returns an empty array when entity not found.
   *
   * @covers ::renderEntityColumn
   */
  public function testRenderEntityColumnNotFound(): void {
    $entity = $this->createMockEntity('different-uuid');
    $otherEntity = $this->createMockEntity('test-uuid-456');
    $style = $this->createMockStylePluginWithResult([
      'columns' => ['title' => 'title'],
      'info' => ['title' => ['separator' => '']],
    ], $otherEntity);

    $result = $this->renderer->renderEntityColumn($style, $entity, 'title');

    $this->assertIsArray($result);
    $this->assertEmpty($result);
  }

  /**
   * Creates a mock style plugin.
   *
   * @param array $options
   *   The style options.
   *
   * @return \Drupal\views\Plugin\views\style\StylePluginBase
   *   The mocked style plugin.
   *
   * @throws \PHPUnit\Framework\MockObject\Exception
   */
  protected function createMockStylePlugin(array $options): StylePluginBase {
    $style = $this->createMock(StylePluginBase::class);
    $style->options = $options;
    return $style;
  }

  /**
   * Creates a mock style plugin with a view result.
   *
   * @param array $options
   *   The style options.
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity to include in the result.
   *
   * @return \Drupal\views\Plugin\views\style\StylePluginBase
   *   The mocked style plugin with a view result.
   *
   * @throws \PHPUnit\Framework\MockObject\Exception
   */
  protected function createMockStylePluginWithResult(array $options, ContentEntityInterface $entity): StylePluginBase {
    $row = new ResultRow();
    $row->_entity = $entity;
    $row->index = 0;

    $view = $this->createMock(ViewExecutable::class);
    $view->result = [$row];

    $style = $this->createMock(StylePluginBase::class);
    $style->options = $options;
    $style->view = $view;

    return $style;
  }

  /**
   * Creates a mock content entity.
   *
   * @param string $uuid
   *   The entity UUID.
   *
   * @return \Drupal\Core\Entity\ContentEntityInterface
   *   The mocked entity.
   *
   * @throws \PHPUnit\Framework\MockObject\Exception
   */
  protected function createMockEntity(string $uuid): ContentEntityInterface {
    $entity = $this->createMock(ContentEntityInterface::class);
    $entity->expects($this->any())
      ->method('uuid')
      ->willReturn($uuid);
    return $entity;
  }

}
