# Image 404 Fallback

A Drupal module that automatically replaces 404 (not found) image requests with a fallback image, improving user experience and preventing broken image displays.

## Description

The Image 404 Fallback module intercepts requests for missing image files and serves a configurable fallback image instead of returning a 404 error. This ensures that broken image links display a fallback rather than a broken image icon, maintaining a cleaner appearance on your site.

## Features

- Automatically detects 404 image requests
- Serves a configurable fallback image
- Supports multiple image formats (JPG, PNG, GIF, WebP, SVG, BMP, ICO, AVIF)
- Includes a default fallback image (SVG)
- Configurable fallback path via admin interface
- Proper HTTP headers and caching
- Handles both direct file requests and image style derivative requests

## Requirements

- Drupal 10.x or 11.x
- PHP 8.1 or higher

## Installation

### Using Composer (Recommended)

If the module is available as a Composer package, install it using:

```bash
composer require drupal/image_404_fallback
```

Then enable the module:

```bash
drush en image_404_fallback -y
```

Or enable it via the Drupal admin interface at `/admin/modules`

## Configuration

1. Navigate to **Configuration > Media > Image 404 Fallback Settings** (`/admin/config/media/image-404-fallback`)
2. Optionally specify a custom fallback image path:
   - Enter an absolute path (e.g., `/var/www/html/placeholders/my-fallback.png`)
   - Or a path relative to the Drupal root (e.g., `sites/default/files/fallback.png`)
3. If left empty, the module will use its default fallback image located at `images/placeholder.svg`

## How It Works

The module uses Symfony event subscribers to intercept HTTP requests:

1. **Exception Event Handler**: Catches 404 exceptions for image requests (priority 250, before Fast 404 handler)
2. **Response Event Handler**: Handles 404 responses for image style derivatives (which return Response objects, not exceptions)
3. **Fallback Serving**: If an image file is not found, the module serves the configured fallback image with appropriate HTTP headers

**Important note:** Private files go through Drupal's access control system. This code only provides a fallback for missing files; it does not bypass access checks. If a private file exists but the user doesn't have access, Drupal's normal access control will still apply.

### Supported Image Extensions

The module recognizes the following image file extensions:
- `.jpg`, `.jpeg`
- `.png`
- `.gif`
- `.webp`
- `.svg`
- `.bmp`
- `.ico`
- `.avif`

## Technical Details

### Event Subscribers

The module subscribes to two Symfony kernel events:
- `KernelEvents::EXCEPTION` (priority 250) - Handles 404 exceptions for direct file requests. Priority is set high to catch 404s before Drupal's Fast 404 handler (priority 200).
- `KernelEvents::RESPONSE` (priority 10) - Handles 404 responses for image style derivatives, which return Response objects rather than throwing exceptions.

### Default Fallback

The module includes a default SVG fallback image at:
```
web/modules/contrib/image_404_fallback/images/placeholder.svg
```

If a custom fallback is not configured, this default will be used.

### HTTP Headers

When serving the fallback, the module sets:
- Appropriate `Content-Type` header based on the fallback image format
- `Cache-Control: public, max-age=3600` for browser caching
- HTTP 200 status code (instead of 404)

## Logging

The module logs the following events:
- **Warning**: When the configured fallback image cannot be found

Note: Detailed logging was removed for production use. The module operates silently in the background, only logging warnings when there are configuration issues.

View logs at `/admin/reports/dblog` or via Drush:
```bash
drush watchdog-show --filter=image_404_fallback
```

## Troubleshooting

### Fallback Not Showing

1. **Check file permissions**: Ensure the fallback image file is readable by the web server
2. **Verify path**: Check that the configured path is correct (use absolute paths if relative paths don't work)
3. **Check logs**: Review the watchdog logs for error messages
4. **Clear cache**: Clear Drupal cache after configuration changes:
   ```bash
   drush cr
   ```

### Module Not Intercepting Requests

1. **Verify module is enabled**: Check that the module is enabled
2. **Check file extensions**: Ensure the requested file has a supported image extension
3. **Review event priorities**: The module uses specific event priorities; conflicts with other modules are possible

### Custom Fallback Not Loading

1. **Path format**: Try using an absolute path instead of a relative path
2. **File existence**: Verify the file exists at the specified path
3. **File permissions**: Ensure the web server can read the file
4. **Check logs**: Look for warning messages about the fallback path

## Development

### Module Structure

```
image_404_fallback/
├── composer.json
├── image_404_fallback.info.yml
├── image_404_fallback.module
├── image_404_fallback.routing.yml
├── image_404_fallback.services.yml
├── image_404_fallback.links.menu.yml
├── config/
│   ├── install/
│   │   └── image_404_fallback.settings.yml
│   └── schema/
│       └── image_404_fallback.schema.yml
├── images/
│   └── placeholder.svg
├── tests/
│   └── src/
│       └── Unit/
│           └── EventSubscriber/
│               └── Image404SubscriberTest.php
├── README.md
└── src/
    ├── EventSubscriber/
    │   └── Image404Subscriber.php
    └── Form/
        └── Image404FallbackSettingsForm.php
```

### Testing

The module includes PHPUnit tests. Run them with:

```bash
ddev exec vendor/bin/phpunit -c web/core/phpunit.xml.dist --testsuite unit web/modules/contrib/image_404_fallback/tests/src/Unit/EventSubscriber/Image404SubscriberTest.php
```

### Code Quality

The module follows Drupal coding standards. Check code style with:

```bash
ddev exec vendor/bin/phpcs --standard=phpcs.xml web/modules/contrib/image_404_fallback/
```

### Extending the Module

To customize the fallback logic, you can:
- Override the `Image404Subscriber` class
- Implement custom event subscribers with different priorities
- Add additional configuration options via the settings form

## License

This module is provided as-is for use in this Drupal project.

