<?php

namespace Drupal\image_404_fallback\EventSubscriber;

use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Psr\Log\LoggerInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpFoundation\BinaryFileResponse;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Event\ExceptionEvent;
use Symfony\Component\HttpKernel\Event\ResponseEvent;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;
use Symfony\Component\HttpKernel\KernelEvents;

/**
 * Event subscriber to replace 404 image requests with a fallback.
 */
class Image404Subscriber implements EventSubscriberInterface {

  /**
   * The module name.
   */
  protected const MODULE_NAME = 'image_404_fallback';

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected FileSystemInterface $fileSystem;


  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected LoggerChannelFactoryInterface $loggerFactory;

  /**
   * The logger instance (lazy-loaded).
   *
   * @var \Psr\Log\LoggerInterface|null
   */
  protected ?LoggerInterface $logger = NULL;

  /**
   * Cached module directory path.
   *
   * @var string|null
   */
  protected static ?string $modulePath = NULL;


  /**
   * Supported image extensions.
   *
   * @var array
   */
  protected array $imageExtensions = ['jpg', 'jpeg', 'png', 'gif', 'webp', 'svg', 'bmp', 'ico', 'avif'];

  /**
   * Constructs an Image404Subscriber.
   *
   * @param \Drupal\Core\File\FileSystemInterface $file_system
   *   The file system service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory service.
   */
  public function __construct(FileSystemInterface $file_system, LoggerChannelFactoryInterface $logger_factory) {
    $this->fileSystem = $file_system;
    $this->loggerFactory = $logger_factory;
  }

  /**
   * Gets the logger instance (lazy-loaded).
   *
   * @return \Psr\Log\LoggerInterface
   *   The logger instance.
   */
  protected function getLogger(): LoggerInterface {
    if ($this->logger === NULL) {
      $this->logger = $this->loggerFactory->get('image_404_fallback');
    }
    return $this->logger;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      // High priority to catch 404s before Fast 404 handler (priority 200).
      KernelEvents::EXCEPTION => ['onException', 250],
      // Check responses for 404 image requests (image style derivatives return
      // Response, not exception).
      KernelEvents::RESPONSE => ['onResponse', 10],
    ];
  }

  /**
   * Handles exceptions and replaces 404 image requests with fallback.
   *
   * @param \Symfony\Component\HttpKernel\Event\ExceptionEvent $event
   *   The exception event.
   */
  public function onException(ExceptionEvent $event): void {
    $exception = $event->getThrowable();

    // Only handle 404 exceptions.
    if (!$exception instanceof NotFoundHttpException) {
      return;
    }

    $request = $event->getRequest();
    $path = $request->getPathInfo();
    // Also check the request URI in case pathInfo doesn't have the extension.
    $uri = $request->getRequestUri();
    $uri_path = parse_url($uri, PHP_URL_PATH) ?: $uri;

    // For private files, the file path might be in the query parameter.
    $file_query = $request->query->get('file');
    $check_paths = [$path, $uri_path];
    if ($file_query) {
      $check_paths[] = $file_query;
      // Also check the full path with /system/files/ prefix.
      if (strpos($path, '/system/files') === 0) {
        $check_paths[] = '/system/files/' . $file_query;
      }
    }

    // Check if the request is for an image file.
    $is_image = FALSE;
    foreach ($check_paths as $check_path) {
      if ($this->isImageRequest($check_path)) {
        $is_image = TRUE;
        break;
      }
    }

    if (!$is_image) {
      return;
    }

    $this->servePlaceholder($event);
  }

  /**
   * Handles 404 responses for image requests (e.g., image style derivatives).
   *
   * @param \Symfony\Component\HttpKernel\Event\ResponseEvent $event
   *   The response event.
   */
  public function onResponse(ResponseEvent $event): void {
    // Only handle main requests.
    if (!$event->isMainRequest()) {
      return;
    }

    $response = $event->getResponse();

    // Only handle 404 responses.
    if ($response->getStatusCode() !== Response::HTTP_NOT_FOUND) {
      return;
    }

    $request = $event->getRequest();
    $path = $request->getPathInfo();
    $uri = $request->getRequestUri();
    $uri_path = parse_url($uri, PHP_URL_PATH) ?: $uri;

    // For private files, the file path might be in the query parameter.
    $file_query = $request->query->get('file');
    $check_paths = [$path, $uri_path];
    if ($file_query) {
      $check_paths[] = $file_query;
      // Also check the full path with /system/files/ prefix.
      if (strpos($path, '/system/files') === 0) {
        $check_paths[] = '/system/files/' . $file_query;
      }
    }

    // Check if the request is for an image file.
    $is_image = FALSE;
    foreach ($check_paths as $check_path) {
      if ($this->isImageRequest($check_path)) {
        $is_image = TRUE;
        break;
      }
    }

    if (!$is_image) {
      return;
    }

    $this->servePlaceholderForResponse($event);
  }

  /**
   * Serves the fallback image as a response.
   *
   * @param \Symfony\Component\HttpKernel\Event\ExceptionEvent $event
   *   The event object.
   */
  protected function servePlaceholder($event): void {
    // Get fallback image path.
    $placeholder_path = $this->getPlaceholderImagePath();

    if (!$placeholder_path || !file_exists($placeholder_path)) {
      $this->getLogger()->warning('Placeholder image not found. Path: @path', ['@path' => $placeholder_path ?? 'NULL']);
      return;
    }

    // Create a response with the fallback image.
    $response = new BinaryFileResponse($placeholder_path);
    $response->headers->set('Content-Type', $this->getImageMimeType($placeholder_path));
    $response->setStatusCode(Response::HTTP_OK);
    // Add cache headers.
    $response->headers->set('Cache-Control', 'public, max-age=3600');

    // Set the response - this stops event propagation automatically.
    $event->setResponse($response);
    // Mark the exception as handled to prevent other handlers from
    // processing it.
    $event->stopPropagation();
  }

  /**
   * Serves the fallback image for a ResponseEvent.
   *
   * @param \Symfony\Component\HttpKernel\Event\ResponseEvent $event
   *   The response event.
   */
  protected function servePlaceholderForResponse(ResponseEvent $event): void {
    // Get fallback image path.
    $placeholder_path = $this->getPlaceholderImagePath();

    if (!$placeholder_path || !file_exists($placeholder_path)) {
      $this->getLogger()->warning('Placeholder image not found. Path: @path', ['@path' => $placeholder_path ?? 'NULL']);
      return;
    }

    // Create a response with the fallback image.
    $response = new BinaryFileResponse($placeholder_path);
    $response->headers->set('Content-Type', $this->getImageMimeType($placeholder_path));
    $response->setStatusCode(Response::HTTP_OK);
    // Add cache headers.
    $response->headers->set('Cache-Control', 'public, max-age=3600');

    $event->setResponse($response);
  }

  /**
   * Gets the file path from the request.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   *
   * @return string|null
   *   The file path if it can be determined, NULL otherwise.
   */
  protected function getFilePathFromRequest($request): ?string {
    $path = $request->getPathInfo();
    $uri = $request->getRequestUri();
    $uri_path = parse_url($uri, PHP_URL_PATH) ?: $uri;

    // Try to determine the file path.
    $drupal_root = DRUPAL_ROOT;

    // Check if it's a public file.
    if (strpos($path, '/sites/') === 0 || strpos($uri_path, '/sites/') === 0) {
      $file_path = $drupal_root . ($path ?: $uri_path);
      return $file_path;
    }

    // Check if it's in the web root.
    $web_path = $path ?: $uri_path;
    if (strpos($web_path, '/') === 0) {
      $file_path = $drupal_root . $web_path;
      if (file_exists($file_path)) {
        return $file_path;
      }
    }

    return NULL;
  }

  /**
   * Checks if the request path is for an image file.
   *
   * @param string $path
   *   The request path.
   *
   * @return bool
   *   TRUE if the path appears to be an image request, FALSE otherwise.
   */
  protected function isImageRequest(string $path): bool {
    // Remove query string if present.
    $path = strtok($path, '?');
    $extension = strtolower(pathinfo($path, PATHINFO_EXTENSION));
    return !empty($extension) && in_array($extension, $this->imageExtensions);
  }

  /**
   * Gets the fallback image path.
   *
   * @return string|null
   *   The path to the fallback image, or NULL if not configured.
   */
  protected function getPlaceholderImagePath(): ?string {
    // Use static config access to avoid circular dependency with ConfigFactory.
    // This is an acceptable pattern in Drupal when DI would cause circular
    // dependencies.
    $config = \Drupal::config(self::MODULE_NAME . '.settings');
    $placeholder_path = $config->get('placeholder_path');

    // Resolve configured placeholder path if provided.
    if ($placeholder_path) {
      $resolved_path = $this->resolvePath($placeholder_path);
      if ($resolved_path) {
        return $resolved_path;
      }
    }

    // Fall back to default module placeholder images.
    return $this->getDefaultPlaceholderPath();
  }

  /**
   * Resolves a file path (handles both absolute and relative paths).
   *
   * @param string $path
   *   The path to resolve (can be absolute or relative to Drupal root).
   *
   * @return string|null
   *   The resolved absolute path if the file exists, NULL otherwise.
   */
  protected function resolvePath(string $path): ?string {
    // If it's already an absolute path and exists, return it.
    if (file_exists($path)) {
      return $path;
    }

    // Try relative to Drupal root.
    $drupal_root = DRUPAL_ROOT;
    $absolute_path = $drupal_root . '/' . ltrim($path, '/');
    if (file_exists($absolute_path)) {
      return $absolute_path;
    }

    return NULL;
  }

  /**
   * Gets the default placeholder image path from the module directory.
   *
   * @return string|null
   *   The path to the default placeholder image, or NULL if not found.
   */
  protected function getDefaultPlaceholderPath(): ?string {
    $module_dir = $this->getModulePath();
    if (!$module_dir) {
      return NULL;
    }

    // Try SVG first, then PNG.
    $extensions = ['svg', 'png'];
    foreach ($extensions as $ext) {
      $placeholder_path = $module_dir . '/images/placeholder.' . $ext;
      if (file_exists($placeholder_path)) {
        return $placeholder_path;
      }
    }

    return NULL;
  }

  /**
   * Gets the module directory path.
   *
   * Uses static caching to avoid repeated reflection calls.
   * Uses reflection to avoid circular dependency with ModuleExtensionList.
   *
   * @return string|null
   *   The absolute path to the module directory, or NULL if not found.
   */
  protected function getModulePath(): ?string {
    if (self::$modulePath !== NULL) {
      return self::$modulePath;
    }

    // Use reflection to get module path without circular dependency.
    // The class is in src/EventSubscriber/, so we need to go up 3 levels to
    // get to module root.
    $reflection = new \ReflectionClass($this);
    $class_file = $reflection->getFileName();
    // Go from src/EventSubscriber/Image404Subscriber.php ->
    // src/EventSubscriber/ -> src/ -> module root.
    self::$modulePath = dirname(dirname(dirname($class_file)));

    return self::$modulePath;
  }

  /**
   * Gets the MIME type for an image file.
   *
   * @param string $file_path
   *   The path to the image file.
   *
   * @return string
   *   The MIME type.
   */
  protected function getImageMimeType(string $file_path): string {
    $extension = strtolower(pathinfo($file_path, PATHINFO_EXTENSION));
    $mime_types = [
      'jpg' => 'image/jpeg',
      'jpeg' => 'image/jpeg',
      'png' => 'image/png',
      'gif' => 'image/gif',
      'webp' => 'image/webp',
      'svg' => 'image/svg+xml',
      'bmp' => 'image/bmp',
      'ico' => 'image/x-icon',
      'avif' => 'image/avif',
    ];

    return $mime_types[$extension] ?? 'image/png';
  }

}
