<?php

declare(strict_types=1);

namespace Drupal\Tests\image_404_fallback\Unit\EventSubscriber;

use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\image_404_fallback\EventSubscriber\Image404Subscriber;
use Drupal\Tests\UnitTestCase;
use Symfony\Component\HttpFoundation\BinaryFileResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Event\ExceptionEvent;
use Symfony\Component\HttpKernel\Event\ResponseEvent;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;
use Symfony\Component\HttpKernel\HttpKernelInterface;
use Symfony\Component\HttpKernel\KernelEvents;

/**
 * @coversDefaultClass \Drupal\image_404_fallback\EventSubscriber\Image404Subscriber
 * @group image_404_fallback
 */
class Image404SubscriberTest extends UnitTestCase {

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface|\Prophecy\Prophecy\ObjectProphecy
   */
  protected $fileSystem;

  /**
   * The logger factory service.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface|\Prophecy\Prophecy\ObjectProphecy
   */
  protected $loggerFactory;

  /**
   * The logger channel.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface|\Prophecy\Prophecy\ObjectProphecy
   */
  protected $loggerChannel;

  /**
   * The HTTP kernel.
   *
   * @var \Symfony\Component\HttpKernel\HttpKernelInterface|\Prophecy\Prophecy\ObjectProphecy
   */
  protected $kernel;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->fileSystem = $this->prophesize(FileSystemInterface::class);
    $this->loggerFactory = $this->prophesize(LoggerChannelFactoryInterface::class);
    $this->loggerChannel = $this->prophesize(LoggerChannelInterface::class);
    $this->kernel = $this->prophesize(HttpKernelInterface::class);

    $this->loggerFactory->get('image_404_fallback')->willReturn($this->loggerChannel->reveal());
    // Allow logger calls but don't require them (some tests may not log).
    // Note: Logger methods are void, so we can't use willReturn().
    // Set up a mock container with config factory for \Drupal::config() calls.
    $config_factory = $this->getConfigFactoryStub([
      'image_404_fallback.settings' => [
        'placeholder_path' => '',
      ],
    ]);
    $container = new ContainerBuilder();
    $container->set('config.factory', $config_factory);
    \Drupal::setContainer($container);
  }

  /**
   * Creates an Image404Subscriber instance.
   *
   * @return \Drupal\image_404_fallback\EventSubscriber\Image404Subscriber
   *   The subscriber instance.
   */
  protected function createSubscriber(): Image404Subscriber {
    return new Image404Subscriber(
      $this->fileSystem->reveal(),
      $this->loggerFactory->reveal()
    );
  }

  /**
   * @covers ::getSubscribedEvents
   */
  public function testGetSubscribedEvents(): void {
    $events = Image404Subscriber::getSubscribedEvents();
    $this->assertArrayHasKey(KernelEvents::EXCEPTION, $events);
    $this->assertArrayHasKey(KernelEvents::RESPONSE, $events);
    $this->assertEquals(['onException', 250], $events[KernelEvents::EXCEPTION]);
    $this->assertEquals(['onResponse', 10], $events[KernelEvents::RESPONSE]);
  }

  /**
   * @covers ::onException
   * @dataProvider providerTestOnExceptionNonImage
   */
  public function testOnExceptionNonImage(string $path, \Throwable $exception): void {
    $subscriber = $this->createSubscriber();
    $request = Request::create($path);
    $event = new ExceptionEvent(
      $this->kernel->reveal(),
      $request,
      HttpKernelInterface::MAIN_REQUEST,
      $exception
    );

    $subscriber->onException($event);

    $this->assertNull($event->getResponse());
  }

  /**
   * Data provider for testOnExceptionNonImage.
   */
  public static function providerTestOnExceptionNonImage(): array {
    return [
      'non-404 exception' => [
        '/test.html',
        new \RuntimeException('Test exception'),
      ],
      '404 for HTML file' => [
        '/test.html',
        new NotFoundHttpException(),
      ],
      '404 for text file' => [
        '/test.txt',
        new NotFoundHttpException(),
      ],
      '404 for no extension' => [
        '/test',
        new NotFoundHttpException(),
      ],
    ];
  }

  /**
   * @covers ::onException
   */
  public function testOnExceptionImage404(): void {
    $subscriber = $this->createSubscriber();
    $request = Request::create('/test.jpg');
    $event = new ExceptionEvent(
      $this->kernel->reveal(),
      $request,
      HttpKernelInterface::MAIN_REQUEST,
      new NotFoundHttpException()
    );

    // Mock the placeholder image path.
    $placeholder_path = $this->getModulePath() . '/images/placeholder.svg';
    if (!file_exists($placeholder_path)) {
      $this->markTestSkipped('Placeholder image not found');
    }

    $subscriber->onException($event);

    $response = $event->getResponse();
    $this->assertInstanceOf(BinaryFileResponse::class, $response);
    $this->assertEquals(Response::HTTP_OK, $response->getStatusCode());
    $this->assertEquals('image/svg+xml', $response->headers->get('Content-Type'));
    // Cache-Control header may have values in different order.
    $cache_control = $response->headers->get('Cache-Control');
    $this->assertStringContainsString('public', $cache_control);
    $this->assertStringContainsString('max-age=3600', $cache_control);
  }

  /**
   * @covers ::onException
   */
  public function testOnExceptionPrivateFile(): void {
    $subscriber = $this->createSubscriber();
    $request = Request::create('/system/files', 'GET', ['file' => 'test.png']);
    $event = new ExceptionEvent(
      $this->kernel->reveal(),
      $request,
      HttpKernelInterface::MAIN_REQUEST,
      new NotFoundHttpException()
    );

    $placeholder_path = $this->getModulePath() . '/images/placeholder.svg';
    if (!file_exists($placeholder_path)) {
      $this->markTestSkipped('Placeholder image not found');
    }

    $subscriber->onException($event);

    $response = $event->getResponse();
    $this->assertInstanceOf(BinaryFileResponse::class, $response);
  }

  /**
   * @covers ::onResponse
   */
  public function testOnResponseNon404(): void {
    $subscriber = $this->createSubscriber();
    $request = Request::create('/test.jpg');
    $response = new Response('', Response::HTTP_OK);
    $event = new ResponseEvent(
      $this->kernel->reveal(),
      $request,
      HttpKernelInterface::MAIN_REQUEST,
      $response
    );

    $subscriber->onResponse($event);

    $this->assertEquals(Response::HTTP_OK, $event->getResponse()->getStatusCode());
  }

  /**
   * @covers ::onResponse
   */
  public function testOnResponse404Image(): void {
    $subscriber = $this->createSubscriber();
    $request = Request::create('/test.png');
    $response = new Response('', Response::HTTP_NOT_FOUND);
    $event = new ResponseEvent(
      $this->kernel->reveal(),
      $request,
      HttpKernelInterface::MAIN_REQUEST,
      $response
    );

    $placeholder_path = $this->getModulePath() . '/images/placeholder.svg';
    if (!file_exists($placeholder_path)) {
      $this->markTestSkipped('Placeholder image not found');
    }

    $subscriber->onResponse($event);

    $new_response = $event->getResponse();
    $this->assertInstanceOf(BinaryFileResponse::class, $new_response);
    $this->assertEquals(Response::HTTP_OK, $new_response->getStatusCode());
    $this->assertEquals('image/svg+xml', $new_response->headers->get('Content-Type'));
  }

  /**
   * @covers ::onResponse
   */
  public function testOnResponseSubRequest(): void {
    $subscriber = $this->createSubscriber();
    $request = Request::create('/test.jpg');
    $response = new Response('', Response::HTTP_NOT_FOUND);
    $event = new ResponseEvent(
      $this->kernel->reveal(),
      $request,
      HttpKernelInterface::SUB_REQUEST,
      $response
    );

    $subscriber->onResponse($event);

    // Should not modify response for sub-requests.
    $this->assertEquals(Response::HTTP_NOT_FOUND, $event->getResponse()->getStatusCode());
  }

  /**
   * @covers ::isImageRequest
   * @dataProvider providerTestIsImageRequest
   */
  public function testIsImageRequest(string $path, bool $expected): void {
    $subscriber = $this->createSubscriber();
    $reflection = new \ReflectionClass($subscriber);
    $method = $reflection->getMethod('isImageRequest');
    $method->setAccessible(TRUE);

    $result = $method->invoke($subscriber, $path);
    $this->assertEquals($expected, $result);
  }

  /**
   * Data provider for testIsImageRequest.
   */
  public static function providerTestIsImageRequest(): array {
    return [
      'jpg extension' => ['/test.jpg', TRUE],
      'jpeg extension' => ['/test.jpeg', TRUE],
      'png extension' => ['/test.png', TRUE],
      'gif extension' => ['/test.gif', TRUE],
      'webp extension' => ['/test.webp', TRUE],
      'svg extension' => ['/test.svg', TRUE],
      'bmp extension' => ['/test.bmp', TRUE],
      'ico extension' => ['/test.ico', TRUE],
      'avif extension' => ['/test.avif', TRUE],
      'uppercase extension' => ['/test.JPG', TRUE],
      'with query string' => ['/test.jpg?foo=bar', TRUE],
      'html file' => ['/test.html', FALSE],
      'txt file' => ['/test.txt', FALSE],
      'no extension' => ['/test', FALSE],
      'empty path' => ['', FALSE],
    ];
  }

  /**
   * @covers ::getImageMimeType
   * @dataProvider providerTestGetImageMimeType
   */
  public function testGetImageMimeType(string $file_path, string $expected): void {
    $subscriber = $this->createSubscriber();
    $reflection = new \ReflectionClass($subscriber);
    $method = $reflection->getMethod('getImageMimeType');
    $method->setAccessible(TRUE);

    $result = $method->invoke($subscriber, $file_path);
    $this->assertEquals($expected, $result);
  }

  /**
   * Data provider for testGetImageMimeType.
   */
  public static function providerTestGetImageMimeType(): array {
    return [
      'jpg' => ['/test.jpg', 'image/jpeg'],
      'jpeg' => ['/test.jpeg', 'image/jpeg'],
      'png' => ['/test.png', 'image/png'],
      'gif' => ['/test.gif', 'image/gif'],
      'webp' => ['/test.webp', 'image/webp'],
      'svg' => ['/test.svg', 'image/svg+xml'],
      'bmp' => ['/test.bmp', 'image/bmp'],
      'ico' => ['/test.ico', 'image/x-icon'],
      'avif' => ['/test.avif', 'image/avif'],
      'unknown extension' => ['/test.xyz', 'image/png'],
    ];
  }

  /**
   * @covers ::resolvePath
   * @dataProvider providerTestResolvePath
   */
  public function testResolvePath(string $path, bool $should_exist): void {
    $subscriber = $this->createSubscriber();
    $reflection = new \ReflectionClass($subscriber);
    $method = $reflection->getMethod('resolvePath');
    $method->setAccessible(TRUE);

    $result = $method->invoke($subscriber, $path);

    if ($should_exist) {
      $this->assertNotNull($result);
      $this->assertFileExists($result);
    }
    else {
      $this->assertNull($result);
    }
  }

  /**
   * Data provider for testResolvePath.
   */
  public static function providerTestResolvePath(): array {
    // Use a file that definitely exists for testing.
    $test_file = __FILE__;
    $drupal_root = DRUPAL_ROOT;
    $relative_path = str_replace($drupal_root . '/', '', $test_file);

    return [
      'absolute path that exists' => [
        $test_file,
        TRUE,
      ],
      'relative path that exists' => [
        $relative_path,
        TRUE,
      ],
      'non-existent path' => [
        '/path/that/does/not/exist.png',
        FALSE,
      ],
      'non-existent relative path' => [
        'web/modules/custom/image_404_fallback/images/nonexistent.png',
        FALSE,
      ],
    ];
  }

  /**
   * @covers ::getDefaultPlaceholderPath
   */
  public function testGetDefaultPlaceholderPath(): void {
    $subscriber = $this->createSubscriber();
    $reflection = new \ReflectionClass($subscriber);
    $method = $reflection->getMethod('getDefaultPlaceholderPath');
    $method->setAccessible(TRUE);

    $result = $method->invoke($subscriber);

    // Should return a path if placeholder exists, or NULL if not.
    if ($result !== NULL) {
      $this->assertFileExists($result);
      $this->assertStringEndsWith('.svg', $result);
    }
  }

  /**
   * @covers ::getModulePath
   */
  public function testGetModulePath(): void {
    $subscriber = $this->createSubscriber();
    $reflection = new \ReflectionClass($subscriber);
    $method = $reflection->getMethod('getModulePath');
    $method->setAccessible(TRUE);

    // First call should calculate the path.
    $result1 = $method->invoke($subscriber);
    $this->assertNotNull($result1);
    $this->assertDirectoryExists($result1);
    $this->assertStringEndsWith('image_404_fallback', $result1);

    // Second call should use cached value (same result).
    $result2 = $method->invoke($subscriber);
    $this->assertEquals($result1, $result2);
  }

  /**
   * @covers ::getPlaceholderImagePath
   */
  public function testGetPlaceholderImagePathWithConfig(): void {
    // Set up config with a custom placeholder path.
    $config_factory = $this->getConfigFactoryStub([
      'image_404_fallback.settings' => [
        'placeholder_path' => __FILE__,
      ],
    ]);
    $container = new ContainerBuilder();
    $container->set('config.factory', $config_factory);
    \Drupal::setContainer($container);

    $subscriber = $this->createSubscriber();
    $reflection = new \ReflectionClass($subscriber);
    $method = $reflection->getMethod('getPlaceholderImagePath');
    $method->setAccessible(TRUE);

    $result = $method->invoke($subscriber);
    $this->assertEquals(__FILE__, $result);
  }

  /**
   * Gets the module path.
   *
   * @return string
   *   The module path.
   */
  protected function getModulePath(): string {
    return dirname(dirname(dirname(dirname(__DIR__))));
  }

}
