<?php

namespace Drupal\image_blurry_placeholder\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configuration form for the Image Blurry Placeholder module.
 */
class ImageBlurryPlaceholderConfigForm extends ConfigFormBase {

  /**
   * Config settings.
   *
   * @var string
   */
  const SETTINGS = 'image_blurry_placeholder.settings';

  /**
   * Implements getEditableConfigNames().
   */
  protected function getEditableConfigNames() {
    return [
      static::SETTINGS,
    ];
  }

  /**
   * Implements getFormId().
   */
  public function getFormId() {
    return 'image_blurry_placeholder_config_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config(static::SETTINGS);

    $form['use_css_js_blur'] = [
      '#title' => $this->t('Use Javascript to apply/remove the "loaded" class on images. Add CSS blur on images not yet loaded.'),
      '#type' => 'checkbox',
      '#default_value' => $config->get('use_css_js_blur'),
      '#config_target' => static::SETTINGS . ':use_css_js_blur',
    ];

    $form['unmanaged_file_expiration'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Placeholder expiration of unmanaged files'),
      '#description' => $this->t("Placeholders of managed files are automatically regenerated when changed. Placeholders of unmanaged files cannot be automatically regenerated like this and are instead regenerated after a certain time has passed. This setting determines this time period, which should be specified as '30 days', '15 days', '12 hours', etc."),
      '#default_value' => $config->get('unmanaged_file_expiration'),
      '#config_target' => static::SETTINGS . ':unmanaged_file_expiration',
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);

    if (!strtotime(sprintf('+%s', $form_state->getValue('unmanaged_file_expiration')))) {
      $form_state->setErrorByName('unmanaged_file_expiration', $this->t('Invalid expiration time.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->config(static::SETTINGS)
      ->set('use_css_js_blur', $form_state->getValue('use_css_js_blur'))
      ->set('unmanaged_file_expiration', $form_state->getValue('unmanaged_file_expiration'))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
