<?php

namespace Drupal\image_blurry_placeholder;

use Drupal\Core\Cache\Cache;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\Entity\EntityViewDisplay;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\FieldableEntityInterface;
use Drupal\file\FileInterface;
use Drupal\image\Entity\ImageStyle;
use Drupal\image\Plugin\Field\FieldType\ImageItem;
use Drupal\image_blurry_placeholder\Form\ImageBlurryPlaceholderConfigForm;

/**
 * Contains helpful functions for the module.
 */
class ImageBlurryPlaceholderManager {

  /**
   * A list of supported image plugins.
   */
  const SUPPORTED_IMAGE_PLUGINS = [
    'media_responsive_thumbnail',
    'media_thumbnail',
    'image',
    'responsive_image',
    'easy_responsive_images',
  ];

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The cache backend.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected $cache;

  /**
   * Constructs a ImageBlurryPlaceholderManager object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache
   *   The cache backend.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    ConfigFactoryInterface $config_factory,
    CacheBackendInterface $cache
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->configFactory = $config_factory;
    $this->cache = $cache;
  }

  /**
   * Check if a formatter plugin ID is supported for blurry placeholder.
   *
   * @param string $plugin_id
   *   The plugin ID.
   *
   * @return bool
   *   TRUE if the plugin is supported.
   */
  public function isSupportedPlugin(string $plugin_id): bool {
    return in_array($plugin_id, self::SUPPORTED_IMAGE_PLUGINS);
  }

  /**
   * Test if a field should be processed by this module.
   *
   * @param \Drupal\Core\Entity\FieldableEntityInterface $entity
   *   The entity.
   * @param string $view_mode
   *   The view mode.
   * @param string $field_name
   *   The field name.
   *
   * @return bool
   *   TRUE if the field should be processed.
   */
  public function shouldProcessField(FieldableEntityInterface $entity, string $view_mode, string $field_name): bool {
    // Get the field formatter settings.
    $entity_display = EntityViewDisplay::collectRenderDisplay($entity, $view_mode);
    $field_display = $entity_display->getComponent($field_name);

    if (!$this->isSupportedPlugin($field_display['type'])) {
      return FALSE;
    }

    if (empty($field_display['third_party_settings']['image_blurry_placeholder']['use_blurry_placeholder'])) {
      return FALSE;
    }

    return TRUE;
  }

  /**
   * Preprocess an image to add a blurry placeholder.
   *
   * @param array $variables
   *   The variables array.
   * @param string $attributesKey
   *   The key in the variables array where the attributes are stored.
   */
  public function preprocessImage(array &$variables, string $attributesKey = 'attributes'): void {
    if (empty($variables[$attributesKey]['use-blurry-placeholder'])) {
      return;
    }

    // Original image uri (no image style).
    $uri = $variables[$attributesKey]['data-original-url'] ?? NULL;
    if (empty($uri)) {
      return;
    }

    unset($variables[$attributesKey]['data-original-url']);

    $placeholderUri = $this->getPlaceholderImage($uri, $variables);
    if ($placeholderUri === NULL) {
      return;
    }

    if (!isset($variables[$attributesKey]['class'])) {
      $variables[$attributesKey]['class'] = [];
    }
    $variables[$attributesKey]['class'][] = 'img-blurry-placeholder';

    $variables[$attributesKey]['style'] = [
      "background-image: url($placeholderUri);",
    ];

    $variables['#attached']['library'][] = 'image_blurry_placeholder/placeholder.style';
    if ($this->shouldLoadJavascriptLibrary()) {
      $variables['#attached']['library'][] = 'image_blurry_placeholder/placeholder.load';
    }
  }

  /**
   * Get the base64 encoded placeholder image.
   *
   * Uses the blurry placeholder image style to create a much smaller
   * version of the image which will then be stretched to fit the container
   * and blurred with css.
   *
   * @param string $uri
   *   The image uri.
   * @param array $variables
   *   The variables array.
   *
   * @return string|null
   *   The base64 encoded image or NULL if the image could not be encoded.
   */
  public function getPlaceholderImage(string $uri, array $variables = []): ?string {
    if ($file = $this->getFileEntity($uri, $variables)) {
      $cacheKey = 'file:' . $file->id();
      $cacheTags = $file->getCacheTags();
      $expiration = Cache::PERMANENT;
    }
    else {
      $cacheKey = 'uri:' . $uri;
      $cacheTags = [];
      $expiration = strtotime(sprintf('+%s', $this->getUnmanagedFileExpiration()));
    }

    if ($cached = $this->cache->get($cacheKey)) {
      return $cached->data;
    }

    // Use the blurry placeholder image style to create a much smaller
    // version of the image which will then be stretched to fit the container
    // and blurred with css.
    $image_style = ImageStyle::load('blurry_placeholder');
    if ($image_style === NULL) {
      return NULL;
    }

    $blurred_image = $image_style->buildUri($uri);
    $success = $image_style->createDerivative($uri, $blurred_image);
    if ($success === FALSE) {
      return NULL;
    }

    // Encode the image in base64 to speed up loading times.
    $img_binary = fread(fopen($blurred_image, 'r'), filesize($blurred_image));
    $file_type = mime_content_type($blurred_image);
    $data = 'data:' . $file_type . ';base64,' . base64_encode($img_binary);

    $this->cache->set($cacheKey, $data, $expiration, $cacheTags);

    return $data;
  }

  /**
   * Attempt to extract the file entity.
   *
   * @param string $uri
   *   The file uri.
   * @param array $variables
   *   The variables array.
   *
   * @return \Drupal\file\FileInterface|null
   *   The file entity or NULL if no entity is found.
   */
  protected function getFileEntity(string $uri, array $variables): ?FileInterface {
    if (($variables['item'] ?? NULL) instanceof ImageItem) {
      $entity = $variables['item']->entity;
      if ($entity instanceof FileInterface) {
        return $entity;
      }
    }

    $entities = $this->entityTypeManager
      ->getStorage('file')
      ->loadByProperties(['uri' => $uri]);

    return reset($entities) ?: NULL;
  }

  /**
   * Check if the Javascript library should be loaded.
   *
   * @return bool
   *   TRUE if the Javascript library should be loaded.
   */
  public function shouldLoadJavascriptLibrary(): bool {
    return (bool) $this->configFactory
      ->get(ImageBlurryPlaceholderConfigForm::SETTINGS)
      ->get('use_css_js_blur');
  }

  /**
   * Get the expiration time for unmanaged files.
   *
   * @return string
   *   The expiration time.
   */
  public function getUnmanagedFileExpiration(): string {
    return $this->configFactory
      ->get(ImageBlurryPlaceholderConfigForm::SETTINGS)
      ->get('unmanaged_file_expiration');
  }

}
