<?php

declare(strict_types=1);

namespace Drupal\image_field_caption\Plugin\Field\FieldType;

use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Element\Checkboxes;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\TypedData\DataDefinition;
use Drupal\image\Plugin\Field\FieldType\ImageItem;

/**
 * Adds support for image captions.
 *
 * @see \image_field_caption_field_info_alter()
 */
class ImageCaptionItem extends ImageItem {

  /**
   * {@inheritdoc}
   */
  public static function defaultStorageSettings(): array {
    $settings = parent::defaultStorageSettings();
    $settings['default_image']['caption'] = '';
    return $settings;
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultFieldSettings(): array {
    $settings = parent::defaultFieldSettings();
    $settings['default_image']['caption'] = '';
    $settings['caption_field'] = FALSE;
    $settings['caption_field_required'] = FALSE;
    $settings['caption_allowed_formats'] = NULL;
    return $settings;
  }

  /**
   * {@inheritdoc}
   */
  public static function schema(FieldStorageDefinitionInterface $field_definition): array {
    $schema = parent::schema($field_definition);
    $schema['columns']['caption'] = [
      'description' => 'The caption text.',
      'type' => 'text',
      'size' => 'big',
      'not null' => FALSE,
    ];
    $schema['columns']['caption_format'] = [
      'description' => 'The caption format.',
      'type' => 'varchar',
      'length' => 255,
      'not null' => FALSE,
    ];
    return $schema;
  }

  /**
   * {@inheritdoc}
   */
  public static function propertyDefinitions(FieldStorageDefinitionInterface $field_definition): array {
    $properties = parent::propertyDefinitions($field_definition);

    $properties['caption'] = DataDefinition::create('string')
      ->setLabel(new TranslatableMarkup('Caption'))
      ->setDescription(new TranslatableMarkup('Short description of the image displayed underneath the image.'));

    $properties['caption_format'] = DataDefinition::create('filter_format')
      ->setLabel(new TranslatableMarkup('Text format of the caption'));

    return $properties;
  }

  /**
   * {@inheritdoc}
   */
  public function fieldSettingsForm(array $form, FormStateInterface $form_state): array {
    $element = parent::fieldSettingsForm($form, $form_state);

    // Add caption configuration options.
    $settings = $this->getSettings() + static::defaultFieldSettings();
    $weight = $element['title_field_required']['#weight'];
    $element['caption_field'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable <em>Caption</em> field'),
      '#default_value' => $settings['caption_field'],
      '#description' => $this->t('Short description of the image displayed underneath the image.'),
      '#weight' => ++$weight,
    ];
    $visible_states['visible'] = [
      ':input[name="settings[caption_field]"]' => ['checked' => TRUE],
    ];
    $element['caption_field_required'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('<em>Caption</em> field required'),
      '#default_value' => $settings['caption_field_required'],
      '#weight' => ++$weight,
      '#states' => $visible_states,
    ];
    /** @var \Drupal\filter\Plugin\DataType\FilterFormat $caption_format_property */
    $caption_format_property = $this->get('caption_format');
    $element['caption_allowed_formats'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('<em>Caption</em> field allowed text formats'),
      '#options' => $caption_format_property->getPossibleOptions(),
      '#default_value' => $settings['caption_allowed_formats'] ?? [],
      '#description' => $this->t('Select the allowed text formats. If no formats are selected, all available text formats will be displayed to the user.'),
      '#element_validate' => [[static::class, 'validateAllowedFormats']],
      '#weight' => ++$weight,
      '#states' => $visible_states,
    ];

    return $element;
  }

  /**
   * Validation callback for the caption_allowed_formats element.
   */
  public static function validateAllowedFormats(array $element, FormStateInterface $form_state): void {
    $value = Checkboxes::getCheckedCheckboxes($form_state->getValue($element['#parents'])) ?: NULL;
    $form_state->setValueForElement($element, $value);
  }

  /**
   * {@inheritdoc}
   */
  protected function defaultImageForm(array &$element, array $settings): void {
    parent::defaultImageForm($element, $settings);
    $element['default_image']['caption'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Caption'),
      '#description' => $this->t('Short description of the image displayed underneath the image.'),
      '#default_value' => $settings['default_image']['caption'] ?? static::defaultFieldSettings()['default_image']['caption'],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public static function calculateDependencies(FieldDefinitionInterface $field_definition): array {
    // Add explicitly allowed formats as config dependencies.
    $dependencies = parent::calculateDependencies($field_definition);
    if ($allowed_formats = $field_definition->getSetting('caption_allowed_formats')) {
      $formats = \Drupal::entityTypeManager()
        ->getStorage('filter_format')
        ->loadMultiple($allowed_formats);
      foreach ($formats as $format) {
        $dependencies[$format->getConfigDependencyKey()][] = $format->getConfigDependencyName();
      }
    }
    return $dependencies;
  }

}
