<?php

declare(strict_types=1);

namespace Drupal\Tests\image_field_caption\Functional;

use Drupal\Core\Entity\FieldableEntityInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\Field\FieldConfigInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Url;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\field_ui\FieldUI;
use Drupal\image_field_caption\Plugin\Field\FieldType\ImageCaptionItem;
use Drupal\Tests\image\Functional\ImageFieldTestBase;
use Drupal\Tests\TestFileCreationTrait;
use Drupal\user\UserInterface;

/**
 * Tests the functionality of the image_field_caption module.
 */
class ImageFieldCaptionTest extends ImageFieldTestBase {

  use TestFileCreationTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['filter_test', 'entity_test'];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * Tests end-to-end functionality.
   *
   * This test performs the following:
   * - Prepares various content entity types before module installation.
   * - Installs the module and verifies that no schema updates are required.
   * - Tests caption functionality across different entity types and field
   * configurations.
   * - Checks field widget behavior, allowed input formats, and caption data
   * saving.
   * - Verifies that the formatter renders the caption correctly.
   * - Uninstalls the module and ensures the schema remains clean.
   */
  public function test(): void {
    // Create basic image fields for testing.
    $state = $this->container->get('state');
    $entity_type_manager = $this->container->get('entity_type.manager');
    $entity_update_manager = $this->container->get('entity.definition_update_manager');
    $test_field_names = [];
    $base_field_variations = [
      [
        'field_name_length' => 8,
        'caption_allowed_formats' => [
          'plain_text',
          'filter_test',
        ],
      ],
      [
        'field_name_length' => FieldStorageConfig::NAME_MAX_LENGTH,
        'caption_allowed_formats' => NULL,
      ],
    ];
    foreach (['entity_test', 'entity_test_mul'] as $entity_type_id) {
      $base_fields = [];
      foreach ($base_field_variations as $field_variation) {
        $base_field_name = $this->randomMachineName($field_variation['field_name_length']);
        $base_fields[$base_field_name] = BaseFieldDefinition::create('image')
          ->setLabel($base_field_name)
          ->setTranslatable(TRUE)
          ->setDisplayOptions('form', [
            'type' => 'image_image',
            'settings' => [],
          ])
          ->setDisplayConfigurable('form', TRUE)
          ->setDisplayOptions('view', [
            'type' => 'image',
            'settings' => [],
          ])
          ->setDisplayConfigurable('view', TRUE)
          ->setSetting('caption_field', TRUE)
          ->setSetting('caption_field_required', TRUE)
          ->setSetting('caption_allowed_formats', $field_variation['caption_allowed_formats']);
        $test_field_names[$entity_type_id][$entity_type_id][$base_field_name] = $field_variation;
      }
      $state->set("$entity_type_id.additional_base_field_definitions", $base_fields);

      $entity_type = $entity_type_manager->getDefinition($entity_type_id);
      $provider = $entity_type->getProvider();
      foreach ($base_fields as $base_field_name => $base_field) {
        $entity_update_manager->installFieldStorageDefinition($base_field_name, $entity_type_id, $provider, $base_field);
      }
      $entity_update_manager->installEntityType($entity_type);
    }

    // Define combinations of entity types and field configurations
    // to test caption functionality.
    $entity_type_variations = [
      'node',
      'user',
      'entity_test',
      'entity_test_rev',
      'entity_test_with_bundle',
      'entity_test_mul',
      'entity_test_mulrev',
    ];
    $field_variations = [
      [
        'field_name_length' => 8,
        'cardinality' => 1,
        'caption_allowed_formats' => [
          'plain_text',
          'filter_test',
        ],
      ],
      [
        'field_name_length' => FieldStorageConfig::NAME_MAX_LENGTH,
        'cardinality' => 2,
        'caption_allowed_formats' => NULL,
      ],
    ];
    $bundle_count = (int) getenv('IMAGE_FIELD_CAPTION_TEST_BUNDLE_COUNT') ?: 2;

    $permissions = [
      'access content',
      'view test entity',
    ];
    $all_formats = filter_formats();
    foreach ($all_formats as $format) {
      $permissions[] = $format->getPermissionName();
    }
    $this->config('filter.settings')
      ->set('always_show_fallback_choice', TRUE)
      ->save();

    foreach ($entity_type_variations as $entity_type_id) {
      $entity_type = $entity_type_manager->getDefinition($entity_type_id);

      // Collect required permissions.
      $permissions[] = $entity_type->getAdminPermission();
      if ($entity_type->get('field_ui_base_route')) {
        $permissions[] = "administer $entity_type_id fields";
      }

      // Create bundles for testing.
      $bundle_ids = [];
      if ($bundle_entity_type_id = $entity_type->getBundleEntityType()) {
        $bundle_entity_type = $entity_type_manager->getDefinition($bundle_entity_type_id);
        $id_key = $bundle_entity_type->getKey('id');
        $label_key = $bundle_entity_type->getKey('label');
        for ($i = 1; $i <= $bundle_count; $i++) {
          $machine_name = $this->randomMachineName();
          $bundle_values = [$id_key => $machine_name];
          if ($label_key) {
            $bundle_values[$label_key] = $machine_name;
          }
          $bundle = $entity_type_manager->getStorage($bundle_entity_type_id)->create($bundle_values);
          $bundle->save();
          $bundle_id = $bundle->id();
          $bundle_ids[] = $bundle_id;

          if ($entity_type_id === 'node') {
            $permissions[] = "create $bundle_id content";
            $permissions[] = "edit any $bundle_id content";
          }
        }
      }
      else {
        $bundle_ids[] = $entity_type_id;
      }

      // Create config image fields for testing.
      foreach ($bundle_ids as $bundle_id) {
        foreach ($field_variations as $field_variation) {
          $field_config = $this->createImageField(
            $this->randomMachineName($field_variation['field_name_length']),
            $entity_type_id,
            $bundle_id,
            ['cardinality' => $field_variation['cardinality']],
          );
          $test_field_names[$entity_type_id][$bundle_id][$field_config->getName()] = $field_variation;
        }
      }
    }

    // The user entity must be reloaded because adding a new field
    // causes broken field definitions until the entity is reloaded.
    $this->adminUser = $entity_type_manager
      ->getStorage($this->adminUser->getEntityTypeId())
      ->loadUnchanged($this->adminUser->id());
    // Grant all necessary permissions to the current user.
    $this->assertInstanceOf(UserInterface::class, $this->adminUser);
    $this->adminUser
      ->addRole($this->drupalCreateRole(array_filter($permissions)))
      ->save();

    // Install the module and verify that no schema updates are required.
    $module_installer = $this->container->get('module_installer');
    $module_installer->install(['image_field_caption']);
    $this->assertFalse($entity_update_manager->needsUpdates(), 'After installation, entity schema is up to date.');

    // Test functionality for each prepared entity type.
    $display_repository = $this->container->get('entity_display.repository');
    $entity_field_manager = $this->container->get('entity_field.manager');
    $file_storage = $entity_type_manager->getStorage('file');
    $test_images = $this->getTestFiles('image');
    foreach ($test_field_names as $entity_type_id => $bundle_ids) {
      $entity_type = $entity_type_manager->getDefinition($entity_type_id);
      $bundle_key = $entity_type->getKey('bundle');
      $label_key = $entity_type->getKey('label');
      foreach ($bundle_ids as $bundle_id => $field_names) {
        $field_definitions = $entity_field_manager->getFieldDefinitions($entity_type_id, $bundle_id);
        $test_fields = [];

        // Collect updated field definitions.
        foreach ($field_names as $field_name => $field_variation) {
          $field_definition = $field_definitions[$field_name];
          $test_fields[] = ['field_definition' => $field_definition] + $field_variation;
          if ($field_definition instanceof FieldConfigInterface) {
            // Create a new image field after module installation
            // to verify the default caption configuration.
            $test_fields[] = [
              'field_definition' => $this->createImageField(
                $this->randomMachineName($field_variation['field_name_length']),
                $entity_type_id,
                $bundle_id,
                ['cardinality' => $field_variation['cardinality']],
              ),
            ] + $field_variation;
          }
        }

        // Test the collected fields.
        foreach ($test_fields as $test_field) {
          $field_definition = $test_field['field_definition'];
          $this->assertInstanceOf(FieldDefinitionInterface::class, $field_definition);

          // Configure field settings (caption support, allowed formats).
          $allowed_formats = $test_field['caption_allowed_formats'];
          if ($field_definition instanceof FieldConfigInterface) {
            // Configure caption support using the UI if available.
            if ($entity_type->get('field_ui_base_route')) {
              $edit = [
                'settings[caption_field]' => '1',
                'settings[caption_field_required]' => '0',
              ];
              if ($allowed_formats) {
                foreach ($allowed_formats as $allowed_format) {
                  $edit["settings[caption_allowed_formats][$allowed_format]"] = '1';
                }
              }
              $route_parameters = FieldUI::getRouteBundleParameter($entity_type, $bundle_id);
              $route_parameters['field_config'] = $field_definition->id();
              $url = Url::fromRoute("entity.field_config.{$entity_type_id}_field_edit_form", $route_parameters);
              $this->drupalGet($url);
              $this->submitForm($edit, 'Save settings');
            }
            else {
              // Otherwise, configure caption support programmatically.
              $field_definition
                ->setSetting('caption_field', TRUE)
                ->setSetting('caption_field_required', TRUE)
                ->setSetting('caption_allowed_formats', $allowed_formats)
                ->save();
            }
          }

          // Create entity and test widget caption input.
          $field_name = $field_definition->getName();
          $cardinality = $field_definition->getFieldStorageDefinition()->getCardinality();
          $entity_storage = $entity_type_manager->getStorage($entity_type_id);
          if ($entity_type_id === 'node' && $cardinality === 1) {
            // For nodes, use real image upload via AJAX.
            $entity_id = $this->uploadNodeImage($test_images[array_rand($test_images)], $field_name, $bundle_id, $this->randomMachineName());
            $entity = $entity_storage->loadUnchanged($entity_id);
          }
          else {
            // For other entities, create them programmatically with image.
            $entity_values = [];
            if ($bundle_key) {
              $entity_values[$bundle_key] = $bundle_id;
            }
            if ($label_key) {
              $entity_values[$label_key] = $this->randomMachineName();
            }
            for ($delta = 0; $delta < $cardinality; $delta++) {
              $image_file = $file_storage->create((array) $test_images[array_rand($test_images)]);
              $image_file->save();
              $entity_values[$field_name][$delta] = [
                'target_id' => $image_file->id(),
                'alt' => $this->randomMachineName(),
                'title' => $this->randomMachineName(),
              ];
            }
            $entity = $entity_storage->create($entity_values);
            if ($entity instanceof UserInterface) {
              $entity->set('name', $this->randomMachineName());
            }
            $entity->save();
          }

          // Fill in the caption field and save.
          $this->drupalGet($entity->toUrl('edit-form'));
          $assert_session = $this->assertSession();
          $edit = [];
          $build_input_name = fn(int $delta, string $element) => "{$field_name}[$delta][caption][$element]";
          for ($delta = 0; $delta < $cardinality; $delta++) {
            // Check that the caption format selector contains
            // only allowed formats.
            $build_format_selector = fn(string $option) => "select[name=\"{$field_name}[$delta][caption][format]\"] option[value=\"$option\"]";
            if ($allowed_formats) {
              $caption_format = $allowed_formats[array_rand($allowed_formats)];
              foreach ($all_formats as $format_id => $format) {
                if (in_array($format_id, $allowed_formats)) {
                  $assert_session->elementExists('css', $build_format_selector($format_id));
                }
                else {
                  $assert_session->elementNotExists('css', $build_format_selector($format_id));
                }
              }
            }
            else {
              $caption_format = array_rand($all_formats);
              foreach ($all_formats as $format_id => $format) {
                $assert_session->elementExists('css', $build_format_selector($format_id));
              }
            }

            $edit[$build_input_name($delta, 'value')] = $this->randomMachineName();
            $edit[$build_input_name($delta, 'format')] = $caption_format;
          }
          $this->submitForm($edit, 'Save');

          // Reload the entity and check the saved caption values.
          $entity = $entity_storage->loadUnchanged($entity->id());
          $this->assertInstanceOf(FieldableEntityInterface::class, $entity);
          foreach ($entity->get($field_name) as $delta => $field_item) {
            $this->assertInstanceOf(ImageCaptionItem::class, $field_item);
            $this->assertEquals($edit[$build_input_name($delta, 'value')], $field_item->get('caption')->getString());
            $this->assertEquals($edit[$build_input_name($delta, 'format')], $field_item->get('caption_format')->getString());
          }

          // Configure and test the image caption formatter.
          $display_repository
            ->getViewDisplay($entity_type_id, $bundle_id)
            ->setComponent($field_name, [
              'type' => 'image_caption',
              'label' => 'above',
              'settings' => [],
            ])
            ->save();
          $this->drupalGet($entity->toUrl());
          $assert_session = $this->assertSession();
          if ($cardinality > 1) {
            for ($delta = 0; $delta < $cardinality; $delta++) {
              $index = $delta + 1;
              $assert_session->elementTextEquals(
                'css',
                "div > div:nth-child($index) > img + blockquote.image-field-caption",
                $edit[$build_input_name($delta, 'value')],
              );
            }
          }
          else {
            $assert_session->elementTextEquals(
              'css',
              'div > img + blockquote.image-field-caption',
              $edit[$build_input_name(0, 'value')],
            );
          }
        }
      }
    }

    // Uninstall the module and check that no further updates are required.
    $module_installer->uninstall(['image_field_caption']);
    $this->assertFalse($entity_update_manager->needsUpdates(), 'After uninstallation, entity schema is up to date.');
    // And reinstall to check whether the database schema was correctly
    // cleaned up during module uninstallation.
    $module_installer->install(['image_field_caption']);
    $this->assertFalse($entity_update_manager->needsUpdates(), 'After installation, entity schema is up to date.');
  }

}
