<?php

declare(strict_types=1);

namespace Drupal\image_field_caption\Hook;

use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\image_field_caption\Plugin\Field\FieldType\ImageCaptionItem;

/**
 * Hook implementations for image_field_caption.
 */
class ImageFieldCaptionHooks {

  use StringTranslationTrait;

  /**
   * Indicates whether the module is currently being uninstalled.
   *
   * Used to alter behavior in certain hooks.
   */
  public static bool $isUninstall = FALSE;

  /**
   * Implements hook_field_info_alter().
   */
  #[Hook('field_info_alter')]
  public function fieldInfoAlter(array &$info): void {
    if (static::$isUninstall) {
      return;
    }

    // Override the image field type class.
    $info['image']['class'] = ImageCaptionItem::class;

    // Enable translation for the caption property.
    $info['image']['column_groups']['caption'] = [
      'label' => $this->t('Caption'),
      'translatable' => TRUE,
    ];
  }

  /**
   * Implements hook_theme().
   */
  #[Hook('theme')]
  public function theme(array $existing): array {
    $base_name = 'image_formatter';
    return [
      'image_caption_formatter' => [
        // As we extend the default image format, the variables passed to the
        // callback function are the same as the original (image_formatter).
        'base hook' => $base_name,
        'variables' => $existing[$base_name]['variables'] + [
          'caption' => NULL,
        ],
      ],
    ];
  }

  /**
   * Implements hook_config_schema_info_alter().
   */
  #[Hook('config_schema_info_alter')]
  public function configSchemaInfoAlter(array &$definitions): void {
    $definitions['field.field_settings.image']['mapping'] += [
      'caption_field' => [
        'type' => 'boolean',
        'label' => 'Enable Caption field',
      ],
      'caption_field_required' => [
        'type' => 'boolean',
        'label' => 'Caption field required',
      ],
      'caption_allowed_formats' => [
        'type' => 'sequence',
        'label' => 'Caption field allowed text formats',
        'nullable' => TRUE,
        'sequence' => ['type' => 'string'],
      ],
    ];
    $definitions['field_default_image']['mapping']['caption'] = [
      'type' => 'label',
      'label' => 'Caption',
    ];
  }

}
