<?php

declare(strict_types=1);

namespace Drupal\image_field_caption\Hook;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\image\Plugin\Field\FieldWidget\ImageWidget;

/**
 * Hook implementations for image_field_caption.
 */
class ImageFieldCaptionWidgetHooks {

  /**
   * Implements hook_field_widget_single_element_WIDGET_TYPE_form_alter() for image_image.
   */
  #[Hook('field_widget_single_element_image_image_form_alter')]
  public function imageWidgetAlter(array &$element, FormStateInterface $form_state, array $context): void {
    /** @var \Drupal\Core\Field\FieldItemListInterface $items */
    $items = $context['items'];
    $settings = $items->getFieldDefinition()->getSettings();
    if (!empty($settings['caption_field'])) {
      $element['#caption_field_required'] = $settings['caption_field_required'];
      $element['#caption_allowed_formats'] = $settings['caption_allowed_formats'];
      // Use #process to check if the field is empty,
      // as it's not possible in this hook.
      $element['#process'][] = [static::class, 'imageWidgetProcess'];
    }
  }

  /**
   * Process callback for the image_image widget.
   */
  public static function imageWidgetProcess(array $element, FormStateInterface $form_state, array $form): array {
    $item = $element['#value'];
    $element['caption'] = [
      '#title' => new TranslatableMarkup('Caption'),
      '#type' => 'text_format',
      '#default_value' => $item['caption'] ?? '',
      '#format' => $item['caption_format'] ?? NULL,
      '#allowed_formats' => $element['#caption_allowed_formats'],
      '#access' => (bool) $item['fids'],
      '#required' => $element['#caption_field_required'],
      '#element_validate' => [[static::class, 'imageWidgetCaptionValidate']],
    ];
    if ($element['#caption_field_required']) {
      $element['caption']['#required'] = TRUE;
      $element['caption']['#element_validate'][] = [ImageWidget::class, 'validateRequiredFields'];
    }
    return $element;
  }

  /**
   * Validation callback for the caption element.
   */
  public static function imageWidgetCaptionValidate(array $element, FormStateInterface $form_state): void {
    // This check prevents duplicate validation since the text_format element
    // sends properties to the textarea element, triggering this callback twice.
    if ($element['#type'] !== 'text_format') {
      return;
    }
    // Extracts values from the text_format element and assigns them directly
    // to field properties.
    $value = $form_state->getValue($element['#parents']);
    $format_parents = array_slice($element['#parents'], 0, -1);
    $format_parents[] = 'caption_format';
    $form_state->setValue($format_parents, $value['format']);
    $form_state->setValueForElement($element, $value['value']);
  }

}
