<?php

declare(strict_types=1);

namespace Drupal\Tests\image_field_caption\Functional;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Url;
use Drupal\field_ui\FieldUI;
use Drupal\FunctionalJavascriptTests\WebDriverTestBase;
use Drupal\Tests\image\Kernel\ImageFieldCreationTrait;
use Drupal\Tests\TestFileCreationTrait;
use PHPUnit\Framework\Attributes\Group;

/**
 * Tests the UI of the image_field_caption module.
 */
#[Group('image_field_caption')]
class ImageFieldCaptionUiTest extends WebDriverTestBase {

  use TestFileCreationTrait;
  use ImageFieldCreationTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['node', 'filter_test', 'field_ui', 'image'];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'claro';

  /**
   * Tests the UI.
   */
  public function testUi(): void {
    // Create a field before module installation to ensure
    // database schema updates, and that default field settings
    // are correctly applied in forms.
    $entity_type_id = 'node';
    $bundle_id = $this->drupalCreateContentType(['type' => 'page'])->id();
    $field_definition = $this->createImageField($this->randomMachineName(), $entity_type_id, $bundle_id, [
      'cardinality' => 2,
    ]);

    // Install the module.
    $this->container->get('module_installer')->install(['image_field_caption']);
    $this->rebuildContainer();

    // Prepare a user with required permissions.
    $default_format = 'full_html';
    $submitted_format = 'filtered_html';
    $permissions = [
      "administer $entity_type_id fields",
      'access content',
      "create $bundle_id content",
      "edit any $bundle_id content",
      "use text format $default_format",
      "use text format $submitted_format",
    ];
    $user = $this->drupalCreateUser($permissions);
    $this->drupalLogin($user);

    // Configure the field via the settings form to ensure
    // the field settings form extension is implemented correctly.
    $entity_type = $this->container->get('entity_type.manager')->getDefinition($entity_type_id);
    $route_parameters = FieldUI::getRouteBundleParameter($entity_type, $bundle_id);
    $route_parameters['field_config'] = $field_definition->id();
    $url = Url::fromRoute("entity.field_config.{$entity_type_id}_field_edit_form", $route_parameters);
    $this->drupalGet($url);

    // Check if #states are configured correctly.
    $assert_session = $this->assertSession();
    $this->assertFalse($assert_session->fieldExists('settings[caption_field_required]')->isVisible());
    $assert_session->fieldExists('settings[caption_field]')->check();
    $this->assertNotEmpty($assert_session->waitForElementVisible('named', ['field', 'settings[caption_field_required]']));

    // Fill in the AJAX part of the form.
    $test_images = $this->getTestFiles('image');
    $file_system = $this->container->get('file_system');
    $edit_ajax = [
      // Set default values at the field storage level to
      // check if field-level overrides work.
      'files[field_storage_subform_settings_default_image_uuid]' => $file_system->realpath(array_pop($test_images)->uri),
      'field_storage[subform][settings][default_image][caption]' => $this->randomMachineName(),
      'field_storage[subform][settings][default_image][caption_format]' => 'filter_test',
      'files[settings_default_image_uuid]' => $file_system->realpath(array_pop($test_images)->uri),
    ];
    $assert_session = $this->assertSession();
    foreach ($edit_ajax as $input => $value) {
      $assert_session->fieldExists($input)->setValue($value);
      $assert_session->assertWaitOnAjaxRequest();
    }
    // Fill in the rest of the form.
    $default_caption = $this->randomMachineName();
    $edit = [
      'settings[caption_field]' => '1',
      'settings[caption_field_required]' => '1',
      'settings[caption_field_default_format]' => $default_format,
      "settings[caption_field_allowed_formats][$default_format]" => '1',
      "settings[caption_field_allowed_formats][$submitted_format]" => '1',
      'settings[default_image][caption]' => $default_caption,
      'settings[default_image][caption_format]' => $default_format,
    ];
    $this->submitForm($edit, 'Save settings');

    // Set the formatter to test rendering.
    $field_name = $field_definition->getName();
    $this->container->get('entity_display.repository')
      ->getViewDisplay($entity_type_id, $bundle_id)
      ->setComponent($field_name, [
        'type' => 'image_caption',
        'settings' => [],
      ])
      ->save();

    // Check if the empty widget is displayed correctly, caption should be
    // available only after the file is uploaded.
    $add_url = Url::fromRoute('node.add', ['node_type' => $bundle_id]);
    $this->drupalGet($add_url);
    $this->drupalGet($add_url);
    $caption_input = "{$field_name}[0][caption][value]";
    $assert_session = $this->assertSession();
    $assert_session->fieldNotExists($caption_input);

    // Check if default values work.
    $edit = [
      'title[0][value]' => $this->randomMachineName(),
    ];
    $this->submitForm($edit, 'Save');
    $assert_session->elementTextEquals('css', 'img + blockquote.image-field-caption', $default_caption);

    // Check if AJAX file upload works.
    $this->drupalGet($add_url);
    $image_path = $file_system->realpath(array_pop($test_images)->uri);
    $assert_session = $this->assertSession();
    $assert_session->fieldExists("files[{$field_name}_0][]")->attachFile($image_path);
    $assert_session->assertExpectedAjaxRequest(1);

    // Check if allowed text formats and default format settings work.
    $format_input = "{$field_name}[0][caption][format]";
    $this->assertNotEmpty($assert_session->waitForElementVisible('named', ['field', $format_input]));
    $this->htmlOutput();
    $assert_session->fieldValueEquals($format_input, $default_format);
    $assert_session->optionExists($format_input, $submitted_format);
    $assert_session->optionNotExists($format_input, 'filter_test');

    // Check if the caption is saved and displayed.
    $title = $this->randomMachineName();
    $submitted_caption = $this->randomMachineName();
    $edit = [
      'title[0][value]' => $title,
      "{$field_name}[0][alt]" => $this->randomMachineName(),
      $caption_input => $submitted_caption,
      $format_input => $submitted_format,
    ];
    $this->submitForm($edit, 'Save');
    $assert_session = $this->assertSession();
    $assert_session->elementTextEquals('css', 'img + blockquote.image-field-caption', $submitted_caption);

    // Check if the selected format is saved.
    $entity = $this->drupalGetNodeByTitle($title);
    $this->assertInstanceOf(ContentEntityInterface::class, $entity);
    $this->assertEquals($submitted_format, $entity->get($field_name)->first()->get('caption_format')->getValue());
  }

}
