<?php

declare(strict_types=1);

namespace Drupal\Tests\image_field_caption\Kernel;

use Drupal\Core\Entity\FieldableEntityInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\image_field_caption\Plugin\Field\FieldType\ImageCaptionItem;
use Drupal\KernelTests\KernelTestBase;
use Drupal\Tests\image\Kernel\ImageFieldCreationTrait;
use Drupal\Tests\TestFileCreationTrait;
use PHPUnit\Framework\Attributes\Group;

/**
 * Tests database schema updates and its operability after updates.
 */
#[Group('image_field_caption')]
class ImageFieldCaptionSchemaTest extends KernelTestBase {

  use TestFileCreationTrait;
  use ImageFieldCreationTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'node',
    'user',
    'file',
    'filter',
    'field',
    'image',
    'filter_test',
    'entity_test',
    'image_field_caption',
  ];

  /**
   * Contains the tested entity types.
   *
   * @var string[]
   */
  protected array $testedEntityTypes;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('file');
    $this->installEntitySchema('user');
    $this->installSchema('user', 'users_data');
    $this->installSchema('file', 'file_usage');
    $this->installConfig(['filter', 'filter_test']);

    $tested_entity_types = [
      // Uses only base_table and no revisions.
      'entity_test' => ['base_table'],
      // Uses base_table + data_table and no revisions.
      'entity_test_mul' => ['base_table', 'data_table'],
      // Uses only base_table and only revision_table.
      'entity_test_rev' => ['base_table', 'revision_table'],
      // Uses base_table + data_table and revision_table + revision_data_table.
      'entity_test_mulrev' => ['base_table', 'data_table', 'revision_table', 'revision_data_table'],
    ];
    // Ensure that the entity type definitions have not changed and only use
    // the specified tables.
    $entity_type_manager = $this->container->get('entity_type.manager');
    foreach ($tested_entity_types as $entity_type_id => $defined_tables) {
      $tables = array_fill_keys($defined_tables, TRUE) + [
        'base_table' => FALSE,
        'data_table' => FALSE,
        'revision_table' => FALSE,
        'revision_data_table' => FALSE,
      ];
      $entity_type = $entity_type_manager->getDefinition($entity_type_id);
      foreach ($tables as $table => $status) {
        $status_text = $status ? 'defined' : 'undefined';
        $this->assertEquals($status, (bool) $entity_type->get($table), "The '$table' must be $status_text in the '$entity_type_id' entity type.");
      }
    }
    $this->testedEntityTypes = array_keys($tested_entity_types);
  }

  /**
   * Tests database schema updates and its operability after updates.
   */
  public function testSchemaUpdateAndUsage(): void {
    // Create base fields and config fields for entity types with different
    // table variants in the database to test schema updates.
    $state = $this->container->get('state');
    $fields_by_entity_type = [];
    foreach ($this->testedEntityTypes as $entity_type_id) {
      $fields_by_entity_type[$entity_type_id][] = $base_field_name = $this->randomMachineName(48);
      $base_field = BaseFieldDefinition::create('image')->setLabel($base_field_name);
      $state->set("$entity_type_id.additional_base_field_definitions", [$base_field_name => $base_field]);
      $this->installEntitySchema($entity_type_id);

      $fields_by_entity_type[$entity_type_id][] = $this->createImageField(
        $this->randomMachineName(FieldStorageConfig::NAME_MAX_LENGTH),
        $entity_type_id,
        $entity_type_id,
        ['cardinality' => 2],
      )->getName();
    }

    // Install the module and check if the schema was updated successfully.
    $module_installer = $this->container->get('module_installer');
    $module_installer->install(['image_field_caption']);
    $this->assertSchemaUpToDate();

    // Create new fields after installing the module to ensure new fields
    // do not have schema issues.
    foreach ($this->testedEntityTypes as $entity_type_id) {
      $fields_by_entity_type[$entity_type_id][] = $this->createImageField(
        $this->randomMachineName(FieldStorageConfig::NAME_MAX_LENGTH),
        $entity_type_id,
        $entity_type_id,
        ['cardinality' => 2],
      )->getName();
    }

    // Create test entities to check if data is saved and loaded correctly.
    $default_format = filter_default_format();
    $entity_type_manager = $this->container->get('entity_type.manager');
    $entity_field_manager = $this->container->get('entity_field.manager');
    $file_storage = $entity_type_manager->getStorage('file');
    $image_file = $file_storage->create((array) $this->getTestFiles('image')[0]);
    $image_file->save();
    $image_file_id = $image_file->id();
    foreach ($fields_by_entity_type as $entity_type_id => $field_names) {
      $field_definitions = $entity_field_manager->getFieldDefinitions($entity_type_id, $entity_type_id);
      $entity_storage = $entity_type_manager->getStorage($entity_type_id);
      $label_key = $entity_storage->getEntityType()->getKey('label');
      foreach ($field_names as $field_name) {
        // Create entity.
        $field_definition = $field_definitions[$field_name];
        $field_name = $field_definition->getName();
        $cardinality = $field_definition->getFieldStorageDefinition()->getCardinality();
        $entity_values = [];
        if ($label_key) {
          $entity_values[$label_key] = $this->randomMachineName();
        }
        for ($delta = 0; $delta < $cardinality; $delta++) {
          $entity_values[$field_name][$delta] = [
            'target_id' => $image_file_id,
            'alt' => $this->randomMachineName(),
            'title' => $this->randomMachineName(),
            'caption' => $this->randomMachineName(),
            'caption_format' => $default_format,
          ];
        }
        $entity = $entity_storage->create($entity_values);
        $entity->save();

        // Check if data is saved.
        $entity = $entity_storage->loadUnchanged($entity->id());
        $this->assertInstanceOf(FieldableEntityInterface::class, $entity);
        foreach ($entity->get($field_name) as $delta => $field_item) {
          $item_values = $entity_values[$field_name][$delta];
          $this->assertInstanceOf(ImageCaptionItem::class, $field_item);
          $this->assertEquals($item_values['caption'], $field_item->get('caption')->getString());
          $this->assertEquals($item_values['caption_format'], $field_item->get('caption_format')->getString());
        }
      }
    }

    // Uninstall the module to check if the schema is properly cleaned up.
    $module_installer->uninstall(['image_field_caption']);
    $this->assertSchemaUpToDate();
    // Additionally, reinstall the module to ensure all created columns were
    // actually removed from the database, since installation creates them
    // without checking for existence.
    $module_installer->install(['image_field_caption']);
    $this->assertSchemaUpToDate();
  }

  /**
   * Asserts that schema of tested entity types does not require updates.
   */
  protected function assertSchemaUpToDate(): void {
    $change_list = $this->container->get('entity.definition_update_manager')->getChangeList();
    $changed_types = implode(', ', array_intersect($this->testedEntityTypes, array_keys($change_list)));
    $this->assertEmpty($changed_types, "The schema for the following tested entity types must be up to date: $changed_types");
  }

}
