<?php

declare(strict_types=1);

namespace Drupal\image_field_caption\Hook;

use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Hook\Attribute\ReorderHook;
use Drupal\Core\Hook\Order\OrderAfter;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\image_field_caption\Plugin\Field\FieldType\ImageCaptionItem;

/**
 * Hook implementations for image_field_caption.
 */
class ImageFieldCaptionHooks {

  use StringTranslationTrait;

  /**
   * Indicates whether the module is currently being uninstalled.
   *
   * Used to alter behavior in certain hooks.
   */
  public static bool $isUninstall = FALSE;

  /**
   * Implements hook_field_info_alter().
   */
  #[Hook('field_info_alter')]
  public function fieldInfoAlter(array &$info): void {
    if (static::$isUninstall) {
      return;
    }

    // Override the image field type class.
    $info['image']['class'] = ImageCaptionItem::class;

    // Enable translation for the caption property.
    $info['image']['column_groups']['caption'] = [
      'label' => $this->t('Caption'),
      'translatable' => TRUE,
    ];
  }

  /**
   * Implements hook_theme().
   *
   * @todo Replace the ReorderHook attribute with the 'order'
   *   parameter of the Hook attribute once the minimum supported Drupal version
   *   for the module is ... || ^11.2 or higher.
   */
  #[Hook('theme')]
  #[ReorderHook('theme', self::class, 'theme', new OrderAfter(modules: ['image', 'responsive_image']))]
  public function theme(array $existing): array {
    // As we extend the default image formatters, the variables passed to the
    // callback function are the same as the original.
    $items = [];
    foreach ([
      'image_formatter' => 'image_caption_formatter',
      'responsive_image_formatter' => 'responsive_image_caption_formatter',
    ] as $base_name => $item_name) {
      if (!$base_variables = $existing[$base_name]['variables'] ?? FALSE) {
        continue;
      }
      $items[$item_name] = [
        'base hook' => $base_name,
        'variables' => $base_variables + [
          'caption' => NULL,
        ],
      ];
    }
    return $items;
  }

  /**
   * Implements hook_config_schema_info_alter().
   */
  #[Hook('config_schema_info_alter')]
  public function configSchemaInfoAlter(array &$definitions): void {
    $definitions['field.field_settings.image']['mapping'] += [
      'caption_field' => [
        'type' => 'boolean',
        'label' => 'Enable Caption field',
      ],
      'caption_field_required' => [
        'type' => 'boolean',
        'label' => 'Caption field required',
      ],
      'caption_field_default_format' => [
        'type' => 'string',
        'label' => 'Caption field default text format',
        'nullable' => TRUE,
      ],
      'caption_field_allowed_formats' => [
        'type' => 'sequence',
        'label' => 'Caption field allowed text formats',
        'nullable' => TRUE,
        'sequence' => ['type' => 'string'],
      ],
    ];
    $definitions['field_default_image']['mapping'] += [
      'caption' => [
        'type' => 'label',
        'label' => 'Caption',
      ],
      'caption_format' => [
        'type' => 'string',
        'label' => 'Caption format',
        'nullable' => TRUE,
      ],
    ];
  }

}
