<?php

declare(strict_types=1);

namespace Drupal\image_field_caption\Plugin\Field\FieldType;

use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Element\Checkboxes;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\TypedData\DataDefinition;
use Drupal\filter\Plugin\DataType\FilterFormat;
use Drupal\image\Plugin\Field\FieldType\ImageItem;

/**
 * Adds support for image captions.
 *
 * @see \Drupal\image_field_caption\Hook\ImageFieldCaptionHooks::fieldInfoAlter()
 */
class ImageCaptionItem extends ImageItem {

  /**
   * {@inheritdoc}
   */
  public static function defaultStorageSettings(): array {
    $settings = parent::defaultStorageSettings();
    $settings['default_image'] += [
      'caption' => '',
      'caption_format' => NULL,
    ];
    return $settings;
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultFieldSettings(): array {
    $settings = parent::defaultFieldSettings();
    $settings += [
      'caption_field' => FALSE,
      'caption_field_required' => FALSE,
      'caption_field_default_format' => NULL,
      'caption_field_allowed_formats' => NULL,
      'caption_field_allowed_widgets' => ['image_image'],
    ];
    $settings['default_image'] += [
      'caption' => '',
      'caption_format' => NULL,
    ];
    return $settings;
  }

  /**
   * {@inheritdoc}
   */
  public static function schema(FieldStorageDefinitionInterface $field_definition): array {
    $schema = parent::schema($field_definition);
    $schema['columns']['caption'] = [
      'description' => 'The caption text.',
      'type' => 'text',
      'size' => 'big',
      'not null' => FALSE,
    ];
    $schema['columns']['caption_format'] = [
      'description' => 'The caption format.',
      'type' => 'varchar',
      'length' => 255,
      'not null' => FALSE,
    ];
    return $schema;
  }

  /**
   * {@inheritdoc}
   */
  public static function propertyDefinitions(FieldStorageDefinitionInterface $field_definition): array {
    $properties = parent::propertyDefinitions($field_definition);

    $properties['caption'] = DataDefinition::create('string')
      ->setLabel(new TranslatableMarkup('Caption'))
      ->setDescription(new TranslatableMarkup('Short description of the image displayed underneath the image.'));

    $properties['caption_format'] = DataDefinition::create('filter_format')
      ->setLabel(new TranslatableMarkup('Text format of the caption'));

    return $properties;
  }

  /**
   * {@inheritdoc}
   */
  public function fieldSettingsForm(array $form, FormStateInterface $form_state): array {
    $element = parent::fieldSettingsForm($form, $form_state);

    // Merge settings with defaults to avoid "Undefined index" errors for
    // new settings on existing config fields.
    $settings = $this->getSettings() + static::defaultFieldSettings();

    // Add caption configuration options.
    $weight = $element['title_field_required']['#weight'];
    $element['caption_field'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable <em>Caption</em> field'),
      '#default_value' => $settings['caption_field'],
      '#description' => $this->t('Short description of the image displayed underneath the image.'),
      '#weight' => ++$weight,
    ];
    $visible_states['visible'] = [
      ':input[name="settings[caption_field]"]' => ['checked' => TRUE],
    ];
    $element['caption_field_required'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('<em>Caption</em> field required'),
      '#default_value' => $settings['caption_field_required'],
      '#weight' => ++$weight,
      '#states' => $visible_states,
    ];
    $format_options = $this->getCaptionFormatProperty()->getPossibleOptions();
    $element['caption_field_default_format'] = [
      '#type' => 'select',
      '#title' => $this->t('<em>Caption</em> field default text format'),
      '#options' => $format_options,
      '#default_value' => $settings['caption_field_default_format'],
      '#element_validate' => [[static::class, 'validateToNull']],
      '#empty_value' => '',
      '#empty_option' => $this->t('- Default -'),
      '#weight' => ++$weight,
      '#states' => $visible_states,
    ];
    $element['caption_field_allowed_formats'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('<em>Caption</em> field allowed text formats'),
      '#options' => $format_options,
      '#default_value' => $settings['caption_field_allowed_formats'] ?? [],
      '#description' => $this->t('Select the allowed text formats. If no formats are selected, all available text formats will be displayed to the user.'),
      '#element_validate' => [[static::class, 'validateCheckboxes'], [static::class, 'validateToNull']],
      '#validate_to_null' => TRUE,
      '#weight' => ++$weight,
      '#states' => $visible_states,
    ];
    $element['caption_field_allowed_widgets'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('<em>Caption</em> field allowed widgets'),
      '#options' => \Drupal::service('plugin.manager.field.widget')->getOptions('image'),
      '#default_value' => $settings['caption_field_allowed_widgets'],
      '#description' => $this->t('Select the widgets where the caption field is available.'),
      '#element_validate' => [[static::class, 'validateCheckboxes']],
      '#weight' => ++$weight,
      '#states' => $visible_states,
    ];

    return $element;
  }

  /**
   * Validation callback that converts empty values to NULL.
   */
  public static function validateToNull(array $element, FormStateInterface $form_state): void {
    $value = $form_state->getValue($element['#parents']) ?: NULL;
    $form_state->setValueForElement($element, $value);
  }

  /**
   * Validation callback that leaves only checked checkbox keys.
   */
  public static function validateCheckboxes(array $element, FormStateInterface $form_state): void {
    $value = Checkboxes::getCheckedCheckboxes($form_state->getValue($element['#parents']));
    $form_state->setValueForElement($element, $value);
  }

  /**
   * {@inheritdoc}
   */
  protected function defaultImageForm(array &$element, array $settings): void {
    parent::defaultImageForm($element, $settings);

    // Merge settings with defaults to avoid "Undefined index" errors for
    // new settings on existing config fields.
    $settings['default_image'] += static::defaultFieldSettings()['default_image'];

    $element['default_image']['caption'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Caption'),
      '#description' => $this->t('Short description of the image displayed underneath the image.'),
      '#default_value' => $settings['default_image']['caption'],
    ];
    $element['default_image']['caption_format'] = [
      '#type' => 'select',
      '#title' => $this->t('Caption format'),
      '#description' => $this->t('Text format of the caption.'),
      '#default_value' => $settings['default_image']['caption_format'] ?? '',
      '#options' => $this->getCaptionFormatProperty()->getPossibleOptions(),
      '#empty_value' => '',
      '#empty_option' => $this->t('- Default -'),
      '#element_validate' => [[static::class, 'validateToNull']],
    ];
  }

  /**
   * Returns an instance of the caption format property.
   *
   * @return \Drupal\filter\Plugin\DataType\FilterFormat
   *   The property instance.
   */
  protected function getCaptionFormatProperty(): FilterFormat {
    return $this->get('caption_format');
  }

  /**
   * {@inheritdoc}
   */
  public static function calculateDependencies(FieldDefinitionInterface $field_definition): array {
    // Add explicitly allowed formats as config dependencies.
    $dependencies = parent::calculateDependencies($field_definition);
    $format_ids = $field_definition->getSetting('caption_field_allowed_formats') ?? [];
    if ($default_format_id = $field_definition->getSetting('default_image')['caption_format'] ?? FALSE) {
      $format_ids[] = $default_format_id;
    }
    foreach (\Drupal::entityTypeManager()->getStorage('filter_format')->loadMultiple($format_ids) as $format) {
      $dependencies[$format->getConfigDependencyKey()][] = $format->getConfigDependencyName();
    }
    return $dependencies;
  }

  /**
   * {@inheritdoc}
   */
  public static function calculateStorageDependencies(FieldStorageDefinitionInterface $field_definition): array {
    $dependencies = parent::calculateStorageDependencies($field_definition);
    if ($default_format_id = $field_definition->getSetting('default_image')['caption_format'] ?? FALSE) {
      $format = \Drupal::entityTypeManager()->getStorage('filter_format')->load($default_format_id);
      $dependencies[$format->getConfigDependencyKey()][] = $format->getConfigDependencyName();
    }
    return $dependencies;
  }

}
