<?php

declare(strict_types=1);

namespace Drupal\Tests\image_field_caption\Functional;

use Drupal\Core\Entity\FieldableEntityInterface;
use Drupal\Core\Entity\RevisionableInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\Site\Settings;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\image_field_caption\Plugin\Field\FieldType\ImageCaptionItem;
use Drupal\Tests\BrowserTestBase;
use Drupal\Tests\image\Kernel\ImageFieldCreationTrait;
use Drupal\Tests\TestFileCreationTrait;
use Drupal\Tests\UpdatePathTestTrait;
use Drupal\user\UserInterface;
use PHPUnit\Framework\Attributes\Group;

/**
 * Tests update path for the image_field_caption module.
 */
#[Group('image_field_caption')]
class ImageFieldCaptionUpdateTest extends BrowserTestBase {

  use ImageFieldCreationTrait;
  use TestFileCreationTrait;
  use UpdatePathTestTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['node', 'image', 'filter_test', 'entity_test'];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * Tests update hook 10401.
   *
   * @see \image_field_caption_update_10401()
   */
  public function testUpdate10401(): void {
    // Creates legacy tables for testing data migration.
    // The image_field_caption_revision table existed in the schema but was
    // never used by the module. It is created here for completeness, but
    // ignored during assertions.
    $tables['image_field_caption'] = [
      'description' => 'The base table for the image_field_caption module.',
      'fields' => [
        'entity_type' => [
          'description' => 'The entity type attached to this caption',
          'type' => 'varchar',
          'length' => 128,
          'not null' => TRUE,
          'default' => '',
        ],
        'bundle' => [
          'description' => 'The bundle attached to this caption',
          'type' => 'varchar',
          'length' => 128,
          'not null' => TRUE,
          'default' => '',
        ],
        'field_name' => [
          'description' => 'The field name attached to this caption',
          'type' => 'varchar',
          'length' => 32,
          'not null' => TRUE,
          'default' => '',
        ],
        'entity_id' => [
          'description' => 'The entity id attached to this caption',
          'type' => 'int',
          'unsigned' => TRUE,
          'not null' => TRUE,
        ],
        'revision_id' => [
          'description' => 'The entity id attached to this caption, or NULL if the entity type is not versioned',
          'type' => 'int',
          'unsigned' => TRUE,
          'not null' => FALSE,
        ],
        'language' => [
          'description' => 'The language attached to this caption',
          'type' => 'varchar',
          'length' => 32,
          'not null' => TRUE,
          'default' => '',
        ],
        'delta' => [
          'description' => 'The sequence number for this caption, used for multi-value fields',
          'type' => 'int',
          'unsigned' => TRUE,
          'not null' => TRUE,
        ],
        'caption' => [
          'description' => 'The caption text.',
          'type' => 'text',
          'not null' => FALSE,
        ],
        'caption_format' => [
          'description' => 'The caption format.',
          'type' => 'varchar',
          'length' => 255,
          'not null' => FALSE,
        ],
      ],
      'indexes' => [
        'entity_type' => ['entity_type'],
        'bundle' => ['bundle'],
        'entity_id' => ['entity_id'],
        'revision_id' => ['revision_id'],
        'language' => ['language'],
      ],
      'primary key' => [
        'entity_type',
        'field_name',
        'entity_id',
        'language',
        'delta',
      ],
    ];
    $tables['image_field_caption_revision'] = [
      'description' => 'The revision table for the image_field_caption module.',
    ] + $tables['image_field_caption'];
    $database = $this->container->get('database');
    $db_schema = $database->schema();
    foreach ($tables as $name => $table) {
      $db_schema->createTable($name, $table);
    }

    // Creates base fields for testing migration from base entity fields.
    // This is needed to verify that the update hook correctly detects and
    // processes fields stored in the entity type's base_table or data_table,
    // since base fields do not use dedicated field tables.
    // * Fields with various name lengths are used because Drupal hashes long
    // field names to derive database table names. This ensures the update
    // properly detects hashed table names.
    $state = $this->container->get('state');
    $entity_type_manager = $this->container->get('entity_type.manager');
    $entity_update_manager = $this->container->get('entity.definition_update_manager');
    foreach (['entity_test', 'entity_test_mul'] as $entity_type_id) {
      $base_fields = [];
      foreach ([8, FieldStorageConfig::NAME_MAX_LENGTH] as $field_name_length) {
        $base_field_name = $this->randomMachineName($field_name_length);
        $base_fields[$base_field_name] = BaseFieldDefinition::create('image')
          ->setLabel($base_field_name)
          ->setTranslatable(TRUE)
          ->setSetting('image_field_caption_test', TRUE);
      }
      $state->set("$entity_type_id.additional_base_field_definitions", $base_fields);

      $entity_type = $entity_type_manager->getDefinition($entity_type_id);
      $provider = $entity_type->getProvider();
      foreach ($base_fields as $base_field_name => $base_field) {
        $entity_update_manager->installFieldStorageDefinition($base_field_name, $entity_type_id, $provider, $base_field);
      }
      $entity_update_manager->installEntityType($entity_type);
    }

    // Prepares various test scenarios with different entity and field types.
    // Includes real entity types like 'node' and 'user', which are widely used
    // and must be explicitly verified. The remaining types are synthetic
    // variations from entity_test modules used to test combinations of
    // multilingual support, revision support, and bundling. This ensures the
    // update hook behaves correctly across all storage and config scenarios.
    $entity_type_variations = [
      'node',
      'user',
      'entity_test',
      'entity_test_rev',
      'entity_test_with_bundle',
      'entity_test_mul',
      'entity_test_mulrev',
      'entity_test_no_bundle',
      'entity_test_mul_with_bundle',
    ];
    // * Fields with various name lengths are used because Drupal hashes long
    // field names to derive database table names. This ensures the update
    // properly detects hashed table names.
    $config_field_variations = [
      ['field_name_length' => 8, 'cardinality' => 1],
      ['field_name_length' => FieldStorageConfig::NAME_MAX_LENGTH, 'cardinality' => 2],
    ];
    // Entity count controls how many entities are created per field and bundle.
    // It defaults to 1.5 times the entity update batch size (usually 50) to
    // ensure that the update process handles multiple batches during testing.
    // This helps verify that batch processing and pagination in the update hook
    // work correctly and no data is skipped or lost.
    // Entity and bundle counts can be adjusted via environment variables.
    // Used for local testing only.
    $entity_count = (int) getenv('IMAGE_FIELD_CAPTION_TEST_ENTITY_COUNT') ?: Settings::get('entity_update_batch_size', 50) * 1.5;
    $bundle_count = (int) getenv('IMAGE_FIELD_CAPTION_TEST_BUNDLE_COUNT') ?: 2;

    // Creates bundles, fields, and data for each tested entity type.
    $entity_field_manager = $this->container->get('entity_field.manager');
    $file_storage = $entity_type_manager->getStorage('file');
    $test_images = $this->getTestFiles('image');
    $invalid_format = 'invalid_format';
    $all_formats = filter_formats();
    $default_format = array_key_first($all_formats);
    $expected_values_by_entity = [];
    foreach ($entity_type_variations as $entity_type_id) {
      // Creates bundles for entities that support them.
      $entity_type = $entity_type_manager->getDefinition($entity_type_id);
      if ($bundle_entity_type_id = $entity_type->getBundleEntityType()) {
        $bundle_entity_type = $entity_type_manager->getDefinition($bundle_entity_type_id);
        $id_key = $bundle_entity_type->getKey('id');
        $label_key = $bundle_entity_type->getKey('label');
        $bundle_ids = [];
        for ($i = 1; $i <= $bundle_count; $i++) {
          $machine_name = $this->randomMachineName();
          $bundle_values = [$id_key => $machine_name];
          if ($label_key) {
            $bundle_values[$label_key] = $machine_name;
          }
          $bundle = $entity_type_manager->getStorage($bundle_entity_type_id)->create($bundle_values);
          $bundle->save();
          $bundle_ids[] = $bundle->id();
        }
      }
      else {
        $bundle_ids = [$entity_type_id];
      }

      // Collects previously defined base fields for this entity type.
      $base_fields = [];
      foreach ($entity_field_manager->getBaseFieldDefinitions($entity_type_id) as $base_field) {
        if ($base_field->getSetting('image_field_caption_test')) {
          $base_fields[] = $base_field;
        }
      }

      $bundle_key = $entity_type->getKey('bundle');
      $label_key = $entity_type->getKey('label');
      $entity_storage = $entity_type_manager->getStorage($entity_type_id);
      foreach ($bundle_ids as $bundle_id) {
        // Creates config fields for each bundle and prepares all fields
        // for testing. Config fields are a common and important case,
        // they always have a dedicated database table,
        // so testing base fields alone is not enough.
        $test_fields = $base_fields;
        foreach ($config_field_variations as ['field_name_length' => $field_name_length, 'cardinality' => $cardinality]) {
          $test_fields[] = $this->createImageField(
            $this->randomMachineName($field_name_length),
            $entity_type_id,
            $bundle_id,
            ['cardinality' => $cardinality],
          );
        }

        // Creates field data in both the entity and the legacy table.
        foreach ($test_fields as $test_field) {
          $field_name = $test_field->getName();
          $cardinality = $test_field->getFieldStorageDefinition()->getCardinality();
          for ($i = 1; $i <= $entity_count; $i++) {
            // Create en entity.
            $entity_values = [];
            if ($bundle_key) {
              $entity_values[$bundle_key] = $bundle_id;
            }
            if ($label_key) {
              $entity_values[$label_key] = $this->randomMachineName();
            }
            for ($delta = 0; $delta < $cardinality; $delta++) {
              $image_file = $file_storage->create((array) $test_images[array_rand($test_images)]);
              $image_file->save();
              $entity_values[$field_name][$delta] = [
                'target_id' => $image_file->id(),
                'alt' => $this->randomMachineName(),
                'title' => $this->randomMachineName(),
              ];
            }
            $entity = $entity_storage->create($entity_values);
            if ($entity instanceof UserInterface) {
              $entity->set('name', $this->randomMachineName());
            }
            $entity->save();

            // Create database rows.
            $entity_id = $entity->id();
            $entity = $entity_storage->loadUnchanged($entity_id);
            $database_values = [
              'entity_type' => $entity_type_id,
              'bundle' => $bundle_id,
              'field_name' => $field_name,
              'entity_id' => $entity_id,
              'revision_id' => $entity instanceof RevisionableInterface ? $entity->getRevisionId() ?? $entity_id : $entity_id,
              'language' => $entity->language()->getId(),
            ];
            $test_invalid_format = TRUE;
            $this->assertInstanceOf(FieldableEntityInterface::class, $entity);
            foreach ($entity->get($field_name) as $delta => $field_item) {
              $caption = $this->randomMachineName();
              // The first item in each field uses an invalid caption_format
              // to test that the update falls back to a valid format properly.
              if ($test_invalid_format) {
                $test_invalid_format = FALSE;
                $actual_caption_format = $invalid_format;
                $expected_caption_format = $default_format;
              }
              else {
                $actual_caption_format = $expected_caption_format = array_rand($all_formats);
              }
              $database
                ->insert('image_field_caption')
                ->fields($database_values + [
                  'delta' => $delta,
                  'caption' => $caption,
                  'caption_format' => $actual_caption_format,
                ])
                ->execute();

              // Builds the expected result set for assertion after the update.
              $expected_values_by_entity[$entity_type_id][$field_name][$entity_id][$delta] = $field_item->getValue() + [
                'caption' => $caption,
                'caption_format' => $expected_caption_format,
              ];
            }
          }
        }
      }
    }
    $this->assertCount(count($entity_type_variations), $expected_values_by_entity);

    // Creates invalid data rows to verify they don't interfere with migration.
    $invalid_entity_type_id = $this->randomMachineName();
    $invalid_bundle_id = $this->randomMachineName();
    $invalid_field_name = $this->randomMachineName();
    $invalid_langcode = $this->randomMachineName(2);
    for ($i = 1; $i <= $entity_count; $i++) {
      $database
        ->insert('image_field_caption')
        ->fields([
          'entity_type' => $invalid_entity_type_id,
          'bundle' => $invalid_bundle_id,
          'field_name' => $invalid_field_name,
          'entity_id' => $i,
          'revision_id' => $i,
          'language' => $invalid_langcode,
          'delta' => rand(0, 10),
          'caption' => $this->randomMachineName(),
          'caption_format' => $invalid_format,
        ])
        ->execute();
    }

    // Runs the update.
    $this->container->get('module_installer')->install(['image_field_caption']);
    $update_registry = $this->container->get('update.update_hook_registry');
    $update_registry->setInstalledVersion('image_field_caption', 10400);
    $this->runUpdates();
    $entity_field_manager->clearCachedFieldDefinitions();
    $entity_type_manager->clearCachedDefinitions();

    // Verifies that all expected data has been migrated from legacy tables to
    // entity fields.
    foreach ($expected_values_by_entity as $entity_type_id => $fields) {
      $entity_storage = $entity_type_manager->getStorage($entity_type_id);
      foreach ($fields as $field_name => $entities) {
        $entity_ids = array_keys($entities);
        $entity_storage->resetCache($entity_ids);
        foreach ($entity_storage->loadMultiple($entity_ids) as $entity_id => $entity) {
          $this->assertInstanceOf(FieldableEntityInterface::class, $entity);

          $expected_items = $entities[$entity_id];
          $actual_items = $entity->get($field_name);
          $this->assertCount(count($expected_items), $actual_items);

          foreach ($actual_items as $delta => $actual_item) {
            $this->assertInstanceOf(ImageCaptionItem::class, $actual_item);
            $this->assertEquals($expected_items[$delta], $actual_item->getValue());
          }
        }
      }
    }
  }

  /**
   * Tests post update hook apply_new_settings.
   *
   * @see \image_field_caption_post_update_apply_new_settings()
   */
  public function testPostUpdateApplyNewSettings(): void {
    $this->container->get('module_installer')->install(['image_field_caption']);
    $this->rebuildContainer();

    $entity_type_id = 'node';
    $bundle_id = $this->drupalCreateContentType()->id();
    $field = $this->createImageField($this->randomMachineName(), $entity_type_id, $bundle_id);

    $this->config($field->getConfigDependencyName())
      ->set('settings.caption_field', TRUE)
      ->set('settings.caption_field_required', TRUE)
      ->clear('settings.caption_field_allowed_formats')
      ->save(TRUE);
    $field_storage = $this->container->get('entity_type.manager')->getStorage('field_config');
    $field_settings = $field_storage->loadUnchanged($field->id())->getSettings();
    $this->assertArrayNotHasKey('caption_field_allowed_formats', $field_settings);

    $post_update = $this->container->get('keyvalue')->get('post_update');
    $existing_updates = $post_update->get('existing_updates', []);
    $update_index = array_search('image_field_caption_post_update_apply_new_settings_3', $existing_updates);
    $this->assertNotFalse($update_index);
    unset($existing_updates[$update_index]);
    $post_update->set('existing_updates', $existing_updates);
    $this->runUpdates();

    $field_settings = $field_storage->loadUnchanged($field->id())->getSettings();
    $this->assertEquals($field_settings['caption_field'], TRUE);
    $this->assertEquals($field_settings['caption_field_required'], TRUE);
    $this->assertArrayHasKey('caption_field_allowed_formats', $field_settings);
  }

}
