<?php

declare(strict_types=1);

namespace Drupal\image_field_caption\Form;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\State\StateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure semantic mode for image captions.
 *
 * @internal
 *
 * @todo Remove as part of https://www.drupal.org/node/3545886
 */
final class ImageFieldCaptionSemanticModeForm extends FormBase {

  /**
   * The state key used to store mode status.
   */
  public const STATE_KEY = 'image_field_caption.semantic_mode';

  /**
   * The state.
   */
  protected StateInterface $state;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    $instance = parent::create($container);
    $instance->state = $container->get('state');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'image_field_caption_semantic_mode_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $args = [
      '@new_tag_1' => '<figure>',
      '@new_tag_2' => '<figcaption>',
      '@old_tag' => '<blockquote>',
      ':issue_url' => 'https://www.drupal.org/node/3545886',
      '@legacy_example' => '<img src="..." alt="...">
<blockquote class="image-field-caption">Example caption</blockquote>',
      '@semantic_example' => '<figure class="caption caption-img image-field-caption" role="group">
  <img src="..." alt="...">
  <figcaption>Example caption</figcaption>
</figure>',
    ];
    $description = $this->t(
      "<p>Switches the caption output between legacy and semantic HTML markup.</p>
      <p>When enabled, captions are rendered using <code>@new_tag_1</code> and <code>@new_tag_2</code>, which provides more meaningful structure and improves accessibility.</p>
      <p>Existing image caption styles will break. The <code>image-field-caption</code> class is now applied to the container element, and the caption no longer inherits any default styles from the <code>@old_tag</code> tag. You will need to update your theme's CSS accordingly.</p>
      <p>If you already override the <code>image-caption-formatter.html.twig</code> template in your theme, enabling this option will not affect you.</p>
      <p>This semantic mode will become the default in the 5.0.0 release, and the legacy mode will be removed. See <a target='_blank' href=':issue_url'>:issue_url</a> for more details.</p>
      <p>Example output:</p>
      <ul>
        <li>Legacy mode:<pre><code>@legacy_example</code></pre></li>
        <li>Semantic mode:<pre><code>@semantic_example</code></pre></li>
      </ul>
      <p>This option exists to preserve backward compatibility and prevent breaking existing styling for existing users.</p>",
      $args,
    );
    $form['semantic_mode'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable semantic mode'),
      '#description' => $description,
    ];
    $form['actions'] = ['#type' => 'actions'];
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save'),
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->state->set(static::STATE_KEY, $form_state->getValue('semantic_mode'));
  }

  /**
   * Checks access to this form.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  public function access(): AccessResultInterface {
    $value = $this->state->get(static::STATE_KEY);
    return AccessResult::allowedIf($value !== NULL)->addCacheTags([static::STATE_KEY]);
  }

}
