<?php

declare(strict_types=1);

namespace Drupal\Tests\image_field_caption\FunctionalJavascript;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Field\FieldConfigInterface;
use Drupal\Core\Url;
use Drupal\field_ui\FieldUI;
use Drupal\FunctionalJavascriptTests\WebDriverTestBase;
use Drupal\Tests\image\Kernel\ImageFieldCreationTrait;
use Drupal\Tests\TestFileCreationTrait;
use PHPUnit\Framework\Attributes\Group;

/**
 * Tests the UI of the image_field_caption module.
 */
#[Group('image_field_caption')]
class ImageFieldCaptionUiTest extends WebDriverTestBase {

  use TestFileCreationTrait;
  use ImageFieldCreationTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['node', 'filter_test', 'field_ui', 'image'];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * Tests the UI.
   */
  public function testUi(): void {
    // Create a field before installing the module to ensure
    // database schema updates are applied and default field settings
    // are correctly set in forms.
    $entity_type_id = 'node';
    $bundle_id = $this->drupalCreateContentType()->id();
    $field_definition = $this->createImageField($this->randomMachineName(), $entity_type_id, $bundle_id, [
      'cardinality' => 2,
    ]);

    // Install the module.
    $this->container->get('module_installer')->install(['image_field_caption']);
    $this->rebuildContainer();

    // Prepare a user with the required permissions.
    $default_format = 'full_html';
    $submitted_format = 'filtered_html';
    $permissions = [
      "administer $entity_type_id fields",
      "administer $entity_type_id display",
      'access content',
      "create $bundle_id content",
      "use text format $default_format",
      "use text format $submitted_format",
    ];
    $user = $this->drupalCreateUser($permissions);
    $this->drupalLogin($user);

    // Configure the field via the settings form to ensure
    // the field settings form extension is implemented correctly.
    $entity_type = $this->container->get('entity_type.manager')->getDefinition($entity_type_id);
    $route_parameters = FieldUI::getRouteBundleParameter($entity_type, $bundle_id);
    $url = Url::fromRoute("entity.field_config.{$entity_type_id}_field_edit_form", $route_parameters + ['field_config' => $field_definition->id()]);
    $this->drupalGet($url);

    // Verify that #states configuration works as expected.
    $assert_session = $this->assertSession();
    $this->assertFalse($assert_session->fieldExists('settings[caption_field_required]')->isVisible());
    $assert_session->fieldExists('settings[caption_field]')->check();
    $this->assertNotEmpty($assert_session->waitForElementVisible('named', ['field', 'settings[caption_field_required]']));

    // Fill in the AJAX-dependent part of the form.
    $test_images = $this->getTestFiles('image');
    $file_system = $this->container->get('file_system');
    $edit_ajax = [
      // Set default values at the field storage level to
      // verify that field-level overrides take effect.
      'files[field_storage_subform_settings_default_image_uuid]' => $file_system->realpath(array_pop($test_images)->uri),
      'field_storage[subform][settings][default_image][caption]' => $this->randomMachineName(),
      'field_storage[subform][settings][default_image][caption_format]' => 'filter_test',
      'files[settings_default_image_uuid]' => $file_system->realpath(array_pop($test_images)->uri),
    ];
    foreach ($edit_ajax as $input => $value) {
      $assert_session->fieldExists($input)->setValue($value);
      $assert_session->assertWaitOnAjaxRequest();
    }
    // Fill in the remaining form fields.
    $field_name = $field_definition->getName();
    $default_caption = $this->randomMachineName();
    $default_widget = 'image_image';
    $edit = [
      'settings[caption_field]' => '1',
      'settings[caption_field_required]' => '1',
      'settings[caption_field_default_format]' => $default_format,
      "settings[caption_field_allowed_formats][$default_format]" => '1',
      "settings[caption_field_allowed_formats][$submitted_format]" => '1',
      "settings[caption_field_allowed_widgets][$default_widget]" => '1',
      'settings[default_image][caption]' => $default_caption,
      'settings[default_image][caption_format]' => $default_format,
    ];
    $this->submitForm($edit, 'Save settings');
    $this->assertTrue($assert_session->waitForText("Saved $field_name configuration."));

    // Set the formatter to test field rendering.
    $url = Url::fromRoute("entity.entity_view_display.$entity_type_id.default", $route_parameters);
    $this->drupalGet($url);
    $assert_session->selectExists("fields[$field_name][type]")->selectOption('image_caption');
    $assert_session->assertWaitOnAjaxRequest();
    $this->submitForm([], 'Save');

    // Verify that an empty widget is displayed correctly:
    // the caption field must be available only after a file is uploaded.
    $add_url = Url::fromRoute('node.add', ['node_type' => $bundle_id]);
    $this->drupalGet($add_url);
    $caption_input = "{$field_name}[0][caption][value]";
    $assert_session->fieldNotExists($caption_input);

    // Verify that default values are applied.
    $edit = [
      'title[0][value]' => $this->randomMachineName(),
    ];
    $this->submitForm($edit, 'Save');
    $assert_session->elementTextEquals('css', 'figure > img + figcaption', $default_caption);

    // Helper function to attach an image to the field.
    $attach_image = function () use ($assert_session, $field_name, $file_system, $test_images) {
      $image_path = $file_system->realpath(array_pop($test_images)->uri);
      $assert_session->fieldExists("files[{$field_name}_0][]")->attachFile($image_path);
      $assert_session->assertExpectedAjaxRequest(1);
    };
    // Verify that the caption field is shown only for allowed widgets.
    $field_definition = $this->container->get('entity_field.manager')->getFieldDefinitions($entity_type_id, $bundle_id)[$field_name];
    $this->assertInstanceOf(FieldConfigInterface::class, $field_definition);
    $field_definition->setSetting('caption_field_allowed_widgets', [])->save();
    $this->drupalGet($add_url);
    $attach_image();
    $assert_session->fieldNotExists($caption_input);
    $field_definition->setSetting('caption_field_allowed_widgets', [$default_widget])->save();

    // Verify that allowed text formats and the default format setting work.
    $this->drupalGet($add_url);
    $attach_image();
    $format_input = "{$field_name}[0][caption][format]";
    $this->assertNotEmpty($assert_session->waitForElementVisible('named', ['field', $format_input]));
    $assert_session->fieldValueEquals($format_input, $default_format);
    $assert_session->optionExists($format_input, $submitted_format);
    $assert_session->optionNotExists($format_input, 'filter_test');
    // Verify that the caption is saved and displayed correctly.
    $title = $this->randomMachineName();
    $submitted_caption = $this->randomMachineName();
    $edit = [
      'title[0][value]' => $title,
      "{$field_name}[0][alt]" => $this->randomMachineName(),
      $caption_input => $submitted_caption,
      $format_input => $submitted_format,
    ];
    $this->submitForm($edit, 'Save');
    $assert_session->elementTextEquals('css', 'figure > img + figcaption', $submitted_caption);
    // Verify caption text format is saved.
    $entity = $this->drupalGetNodeByTitle($title);
    $this->assertInstanceOf(ContentEntityInterface::class, $entity);
    $this->assertEquals($submitted_format, $entity->get($field_name)->first()->get('caption_format')->getValue());
  }

}
