<?php

declare(strict_types=1);

namespace Drupal\Tests\image_field_caption\Kernel;

use Drupal\Component\Utility\Html;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\KernelTests\KernelTestBase;
use Drupal\Tests\image\Kernel\ImageFieldCreationTrait;
use Drupal\Tests\TestFileCreationTrait;
use PHPUnit\Framework\Attributes\Group;

/**
 * Tests integration of responsive_image module with image_field_caption.
 */
#[Group('image_field_caption')]
class ResponsiveImageCaptionTest extends KernelTestBase {

  use TestFileCreationTrait;
  use ImageFieldCreationTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'entity_test',
    'image',
    'file',
    'user',
    'field',
    'filter',
    'breakpoint',
    'image_field_caption',
    'responsive_image',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installConfig(['system']);
    $this->installEntitySchema('entity_test');
    $this->installEntitySchema('file');
    $this->installEntitySchema('user');
    $this->installConfig(['image', 'filter']);
  }

  /**
   * Tests responsive_image_caption formatter.
   */
  public function testResponsiveImageCaptionFormatter(): void {
    // Setup test entity with image field and caption data.
    $entity_type_manager = $this->container->get('entity_type.manager');
    $entity_type_id = $bundle_id = 'entity_test';
    $field_name = $this->createImageField($this->randomMachineName(), $entity_type_id, $bundle_id)->getName();

    $test_images = $this->getTestFiles('image');
    $image_file = $entity_type_manager->getStorage('file')->create((array) $test_images[0]);
    $image_file->save();

    $alt = $this->randomMachineName();
    $caption = $this->randomMachineName();
    $entity = $entity_type_manager->getStorage($entity_type_id)->create([
      $field_name => [
        'target_id' => $image_file->id(),
        'alt' => $alt,
        'caption' => $caption,
        'caption_format' => 'plain_text',
      ],
    ]);
    $this->assertInstanceOf(ContentEntityInterface::class, $entity);

    // Create a responsive image style for the test.
    $responsive_image_style = $entity_type_manager->getStorage('responsive_image_style')->create([
      'id' => $this->randomMachineName(),
      'label' => $this->randomMachineName(),
      'breakpoint_group' => 'responsive_image',
      'fallback_image_style' => 'thumbnail',
      'image_style_mappings' => [
        [
          'breakpoint_id' => 'responsive_image.viewport_sizing',
          'multiplier' => '1x',
          'image_mapping_type' => 'sizes',
          'image_mapping' => [
            'sizes' => '(min-width: 700px) 700px, 100vw',
            'sizes_image_styles' => [
              'large',
              'medium',
            ],
          ],
        ],
      ],
    ]);
    $responsive_image_style->save();

    // Render field with responsive_image_caption formatter.
    $display_options = [
      'type' => 'responsive_image_caption',
      'settings' => [
        'image_link' => '',
        'responsive_image_style' => $responsive_image_style->id(),
      ],
    ];
    $element = $entity->get($field_name)->first()->view($display_options);
    $markup = $this->container->get('renderer')->renderInIsolation($element);

    // Verify that responsive image style was applied.
    $dom = Html::load((string) $markup);
    $xpath = new \DOMXPath($dom);
    $this->assertCount(1, $xpath->query("//figure/img[@srcset and @sizes and @alt='$alt'][following-sibling::figcaption[contains(., '$caption')]]"));
  }

}
