<?php

declare(strict_types=1);

namespace Drupal\image_field_caption\Plugin\Field\FieldFormatter;

use Drupal\Core\Field\EntityReferenceFieldItemListInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\State\StateInterface;
use Drupal\Core\Url;
use Drupal\image_field_caption\Form\ImageFieldCaptionSemanticModeForm;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a base implementation for image formatters with a caption.
 */
trait ImageCaptionFormatterTrait {

  /**
   * The state.
   */
  protected StateInterface $state;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->state = $container->get('state');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state): array {
    $element = parent::settingsForm($form, $form_state);
    if ($this->state->get(ImageFieldCaptionSemanticModeForm::STATE_KEY) === FALSE) {
      $message = $this->t('You are using the legacy caption output mode, which is deprecated and will be removed in 5.0.0. Please switch to semantic output <a href=":url" target="_blank">here</a>.', [
        ':url' => Url::fromRoute('image_field_caption.semantic_mode')->toString(),
      ]);
      $element['semantic_mode'] = [
        '#theme' => 'status_messages',
        '#message_list' => [MessengerInterface::TYPE_WARNING => [$message]],
        '#status_headings' => [MessengerInterface::TYPE_WARNING => $this->t('Legacy caption output mode')],
        '#weight' => -100,
      ];
    }
    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode): array {
    // Overrides the theme to display a caption.
    if ($elements = parent::viewElements($items, $langcode)) {
      $theme = $this->getElementTheme();
      foreach ($elements as &$element) {
        $element['#theme'] = $theme;
      }
    }
    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  protected function getEntitiesToView(EntityReferenceFieldItemListInterface $items, $langcode): array {
    $entities = parent::getEntitiesToView($items, $langcode);
    // @phpstan-ignore property.notFound
    if (\count($entities) === 1 && ($item = $entities[0]->_referringItem) && $item->_is_default) {
      $default_image = $items->getSetting('default_image');
      $item->set('caption', $default_image['caption'] ?? '');
      $item->set('caption_format', $default_image['caption_format'] ?? \filter_default_format());
    }
    return $entities;
  }

  /**
   * Returns the #theme for the element.
   *
   * @return string
   *   The #theme.
   */
  abstract protected function getElementTheme(): string;

}
