<?php

declare(strict_types=1);

namespace Drupal\Tests\image_library_widget\Functional;

use Drupal\Tests\BrowserTestBase;
use Drupal\Tests\image\Kernel\ImageFieldCreationTrait;
use Drupal\Tests\media\Traits\MediaTypeCreationTrait;
use Drupal\Tests\TestFileCreationTrait;
use PHPUnit\Framework\ExpectationFailedException;

/**
 * Tests the Image Library Widget without JavaScript.
 *
 * @group image_library_widget
 */
class ImageLibraryWidgetTest extends BrowserTestBase {

  use ImageFieldCreationTrait;
  use MediaTypeCreationTrait;
  use TestFileCreationTrait;

  /**
   * The ID of the media bundle used for testing.
   *
   * @const string
   */
  protected const TEST_MEDIA_TYPE = 'test_media_type';

  /**
   * Name of the image field used for testing.
   *
   * @const string
   */
  protected const TEST_FIELD_NAME = 'test_field';

  /**
   * Realpath of the image file to use in the test.
   *
   * @const string
   */
  protected string $testFileUri;

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'image_library_widget',
    'entity_test',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->createMediaType(
      'image',
      [
        'id' => static::TEST_MEDIA_TYPE,
        'label' => static::TEST_MEDIA_TYPE,
        'source_configuration' => [
          'source_field' => 'image_library_widget_image',
        ],
      ])
      ->save();

    $this->createImageField(
      field_name: self::TEST_FIELD_NAME,
      entity_type: 'entity_test',
      bundle: 'entity_test',
      field_settings:[
        'file_extensions' => 'png jpeg',
        'alt_field_required' => 0,
        'max_resolution' => '100x100',
        'min_resolution' => '50x50',
      ],
    );

    // Change widget to "image_library_widget".
    \Drupal::service('entity_display.repository')
      ->getFormDisplay('entity_test', 'entity_test')
      ->setComponent(
        self::TEST_FIELD_NAME,
        [
          'type' => 'image_library_widget',
          'settings' => ['media_type_id' => static::TEST_MEDIA_TYPE],
        ]
      )
      ->save();

    $test_image_png = array_reduce(
      $this->getTestFiles('image'),
      function (?object $carry, object $file): ?object {
        if ($carry) {
          return $carry;
        }
        $extension = pathinfo($file->filename, PATHINFO_EXTENSION);
        return $extension === 'png' ? $file : NULL;
      },
    );
    $this->testFileUri = \Drupal::service('file_system')->realpath($test_image_png->uri);

    $this->drupalLogin($this->drupalCreateUser(admin: TRUE));
  }

  /**
   * Tests the image library widget and its validation.
   *
   * @param string|null $moduleToInstall
   *   An additional module to install before testing validation.
   * @param string $errorMessageCssSelector
   *   The CSS selector of the error message.
   *
   * @dataProvider providerWidgetErrorMessages
   */
  public function testWidgetErrorMessages(?string $moduleToInstall, string $errorMessageCssSelector): void {
    if ($moduleToInstall) {
      \Drupal::service('module_installer')->install([$moduleToInstall]);
    }
    $this->drupalGet('/entity_test/add');

    $this->submitForm(
      [
        'name[0][value]' => $this->randomMachineName(),
        'files[' . self::TEST_FIELD_NAME . '_0]' => $this->testFileUri,
      ],
      'Save',
    );

    // We must see the error message.
    $errorMessageElements = $this->getSession()->getPage()->findAll('css', $errorMessageCssSelector);
    $this->assertNotEmpty(
      $errorMessageElements,
      sprintf(
        "Cannot find any error message element matching the CSS selector '%s'",
        $errorMessageCssSelector,
      )
    );

    foreach ($errorMessageElements as $errorElement) {
      if (str_contains($errorElement->getText(), 'The specified file image-test.png could not be uploaded.')) {
        return;
      }
    }

    throw new ExpectationFailedException(sprintf(
      "None of the elements matching the error message CSS selector '%s' contains the expected error message.",
      $errorMessageCssSelector,
    ));
  }

  /**
   * Test data provider for ::testWidgetErrorMessages.
   *
   * @return array
   *   The test cases.
   */
  public static function providerWidgetErrorMessages(): array {
    return [
      'Regular errors' => [
        'moduleToInstall' => NULL,
        'errorMessageCssSelector' => '[role="contentinfo"][aria-label="Error message"] [role="alert"]',
      ],
      'Inline form errors' => [
        'moduleToInstall' => 'inline_form_errors',
        'errorMessageCssSelector' => '.form-item--error-message',
      ],
    ];
  }

}
