<?php

declare(strict_types=1);

namespace Drupal\image_to_media_swapper;

/**
 * Handles batch processing for converting image fields to media entities.
 */
class BatchHandler {

  /**
   * Processes a chunk of entities to convert image fields to media.
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public static function processChunk(string $fieldSelector, array $entity_ids, array &$context): void {
    $group = $context['group'] ?? 'images';
    /** @var \Drupal\image_to_media_swapper\BatchProcessorService $processor */
    $processor = \Drupal::service('image_to_media_swapper.batch_processor_service');
    $entity_type_id = explode('.', $fieldSelector)[0];
    $storage = \Drupal::entityTypeManager()->getStorage($entity_type_id);
    /** @var \Drupal\Core\Entity\FieldableEntityInterface[] $entities */
    $entities = $storage->loadMultiple($entity_ids);

    $converted = $processor->swapImagesToMedia($fieldSelector, $entities);

    if (!isset($context['results']['processed'])) {
      $context['results']['processed'] = 0;
      $context['results']['updated_entities'] = [];
    }

    $entityTypeManager = \Drupal::entityTypeManager();
    $swapRecordStorage = $entityTypeManager->getStorage('media_swap_record');

    foreach ($converted as $entity) {
      $context['results']['processed']++;
      $context['results']['updated_entities'][] = [
        'entity_type' => $entity->getEntityTypeId(),
        'bundle' => $entity->bundle(),
        'id' => $entity->id(),
      ];

      // Create MediaSwapRecord entity.
      try {
        /** @var \Drupal\image_to_media_swapper\Entity\MediaSwapRecordInterface $swapRecord */
        $swapRecord = $swapRecordStorage->create([
          'field_selector' => $fieldSelector,
          'target_entity_type' => $entity->getEntityTypeId(),
          'target_bundle' => $entity->bundle(),
          'target_entity_id' => $entity->id(),
          'batch_category' => $group,
          'processing_status' => 'completed',
          'processed_time' => \Drupal::time()->getRequestTime(),
        ]);
        $swapRecord->save();
      }
      catch (\Exception $e) {
        // Log error but don't fail the batch.
        \Drupal::logger('image_to_media_swapper')->error('Failed to create swap record for entity @id: @error', [
          '@id' => $entity->id(),
          '@error' => $e->getMessage(),
        ]);

        // Create a failed record.
        try {
          $swapRecord = $swapRecordStorage->create([
            'field_selector' => $fieldSelector,
            'target_entity_type' => $entity->getEntityTypeId(),
            'target_bundle' => $entity->bundle(),
            'target_entity_id' => $entity->id(),
            'batch_category' => $group,
            'processing_status' => 'failed',
            'processed_time' => \Drupal::time()->getRequestTime(),
            'error_message' => $e->getMessage(),
          ]);
          $swapRecord->save();
        }
        catch (\Exception $innerE) {
          // If we can't even save the error record, just log it.
          \Drupal::logger('image_to_media_swapper')->critical('Failed to create error swap record: @error', [
            '@error' => $innerE->getMessage(),
          ]);
        }
      }
    }
    $context['message'] = t('Processed chunk of @count entities.', ['@count' => count($converted)]);
  }

  /**
   * Handles the completion of the batch process.
   *
   * @param bool $success
   *   The success status of the batch operation.
   * @param array $results
   *   The results of the batch operation.
   * @param array $operations
   *   The operations that were performed during the batch.
   */
  public static function batchFinished(bool $success, array $results, array $operations): void {
    $messenger = \Drupal::messenger();
    if ($success) {
      $count = $results['processed'] ?? 0;
      if (empty($count)) {
        $messenger->addError(t('No operations were performed.'));
        return;
      }
      $messenger->addStatus(t('Successfully processed @count items.', ['@count' => $count]));
    }
    else {
      $messenger->addError(t('The batch process encountered an error.'));
    }
  }

}
