<?php

declare(strict_types=1);

namespace Drupal\image_to_media_swapper\Entity;

use Drupal\Core\Entity\EntityChangedTrait;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityPublishedTrait;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Field\BaseFieldDefinition;

/**
 * Defines the media swap record entity.
 *
 * @ContentEntityType(
 *   id = "media_swap_record",
 *   label = @Translation("Media Swap Record"),
 *   label_collection = @Translation("Media Swap Records"),
 *   label_singular = @Translation("media swap record"),
 *   label_plural = @Translation("media swap records"),
 *   label_count = @PluralTranslation(
 *     singular = "@count media swap record",
 *     plural = "@count media swap records",
 *   ),
 *   handlers = {
 *     "view_builder" = "Drupal\Core\Entity\EntityViewBuilder",
 *     "list_builder" = "Drupal\image_to_media_swapper\MediaSwapRecordListBuilder",
 *     "views_data" = "Drupal\views\EntityViewsData",
 *     "access" = "Drupal\image_to_media_swapper\MediaSwapRecordAccessControlHandler",
 *     "form" = {
 *       "add" = "Drupal\image_to_media_swapper\Form\MediaSwapRecordForm",
 *       "edit" = "Drupal\image_to_media_swapper\Form\MediaSwapRecordForm",
 *       "delete" = "Drupal\Core\Entity\ContentEntityDeleteForm",
 *     },
 *     "route_provider" = {
 *       "html" = "Drupal\Core\Entity\Routing\AdminHtmlRouteProvider",
 *     }
 *   },
 *   base_table = "media_swap_record",
 *   translatable = FALSE,
 *   admin_permission = "administer image to media swapper",
 *   entity_keys = {
 *     "id" = "id",
 *     "uuid" = "uuid",
 *     "published" = "status",
 *   },
 *   links = {
 *     "add-form" = "/admin/config/media/image-to-media-swapper/records/add",
 *     "canonical" = "/admin/config/media/image-to-media-swapper/records/{media_swap_record}",
 *     "edit-form" = "/admin/config/media/image-to-media-swapper/records/{media_swap_record}/edit",
 *     "delete-form" = "/admin/config/media/image-to-media-swapper/records/{media_swap_record}/delete",
 *     "collection" = "/admin/config/media/image-to-media-swapper/records",
 *   },
 *   field_ui_base_route = "image_to_media_swapper.settings",
 * )
 */
final class MediaSwapRecord extends ContentEntityBase implements MediaSwapRecordInterface {

  use EntityChangedTrait;
  use EntityPublishedTrait;

  /**
   * {@inheritdoc}
   */
  public function getFieldSelector(): string {
    return $this->get('field_selector')->value ?? '';
  }

  /**
   * {@inheritdoc}
   */
  public function setFieldSelector(string $field_selector): MediaSwapRecordInterface {
    $this->set('field_selector', $field_selector);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getTargetEntityType(): string {
    return $this->get('target_entity_type')->value ?? '';
  }

  /**
   * {@inheritdoc}
   */
  public function setTargetEntityType(string $entity_type): MediaSwapRecordInterface {
    $this->set('target_entity_type', $entity_type);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getTargetBundle(): string {
    return $this->get('target_bundle')->value ?? '';
  }

  /**
   * {@inheritdoc}
   */
  public function setTargetBundle(string $bundle): MediaSwapRecordInterface {
    $this->set('target_bundle', $bundle);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getTargetEntityId(): int {
    return (int) ($this->get('target_entity_id')->value ?? 0);
  }

  /**
   * {@inheritdoc}
   */
  public function getTargetEntity(): ?EntityInterface {
    $entity_id = $this->getTargetEntityId();
    if ($entity_id) {
      $entity_type = $this->getTargetEntityType();
      return \Drupal::entityTypeManager()->getStorage($entity_type)->load($entity_id);
    }
    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function setTargetEntityId(int $entity_id): MediaSwapRecordInterface {
    $this->set('target_entity_id', $entity_id);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getBatchCategory(): string {
    return $this->get('batch_category')->value ?? '';
  }

  /**
   * {@inheritdoc}
   */
  public function setBatchCategory(string $category): MediaSwapRecordInterface {
    $this->set('batch_category', $category);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getStatus(): string {
    return $this->get('processing_status')->value ?? 'pending';
  }

  /**
   * {@inheritdoc}
   */
  public function setStatus(string $status): MediaSwapRecordInterface {
    $this->set('processing_status', $status);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getProcessedTime(): int {
    return (int) ($this->get('processed_time')->value ?? 0);
  }

  /**
   * {@inheritdoc}
   */
  public function setProcessedTime(int $timestamp): MediaSwapRecordInterface {
    $this->set('processed_time', $timestamp);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getErrorMessage(): ?string {
    $value = $this->get('error_message')->value;
    return $value !== '' ? $value : NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function setErrorMessage(?string $message): MediaSwapRecordInterface {
    $this->set('error_message', $message ?? '');
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type) {
    $fields = parent::baseFieldDefinitions($entity_type);

    $fields['field_selector'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Field Selector'))
      ->setDescription(t('The field selector in format entity_type.bundle.field_name'))
      ->setRequired(TRUE)
      ->setSetting('max_length', 255)
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => -5,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => -5,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['target_entity_type'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Target Entity Type'))
      ->setDescription(t('The entity type that was processed'))
      ->setRequired(TRUE)
      ->setSetting('max_length', 128)
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => -4,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => -4,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['target_bundle'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Target Bundle'))
      ->setDescription(t('The bundle that was processed'))
      ->setRequired(TRUE)
      ->setSetting('max_length', 128)
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => -3,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => -3,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['target_entity_id'] = BaseFieldDefinition::create('integer')
      ->setLabel(t('Target Entity ID'))
      ->setDescription(t('The ID of the entity that was processed'))
      ->setRequired(TRUE)
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'number_integer',
        'weight' => -2,
      ])
      ->setDisplayOptions('form', [
        'type' => 'number',
        'weight' => -2,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['batch_category'] = BaseFieldDefinition::create('list_string')
      ->setLabel(t('Batch Category'))
      ->setDescription(t('The type of content that was processed'))
      ->setRequired(TRUE)
      ->setSetting('allowed_values', [
        'images' => 'Images',
        'links' => 'File Links',
        'mixed' => 'Mixed Content',
      ])
      ->setDefaultValue('images')
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'list_default',
        'weight' => -1,
      ])
      ->setDisplayOptions('form', [
        'type' => 'options_select',
        'weight' => -1,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['processing_status'] = BaseFieldDefinition::create('list_string')
      ->setLabel(t('Processing Status'))
      ->setDescription(t('The current status of the processing'))
      ->setRequired(TRUE)
      ->setSetting('allowed_values', [
        'pending' => 'Pending',
        'processing' => 'Processing',
        'completed' => 'Completed',
        'failed' => 'Failed',
        'skipped' => 'Skipped',
      ])
      ->setDefaultValue('pending')
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'list_default',
        'weight' => 0,
      ])
      ->setDisplayOptions('form', [
        'type' => 'options_select',
        'weight' => 0,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['processed_time'] = BaseFieldDefinition::create('timestamp')
      ->setLabel(t('Processed Time'))
      ->setDescription(t('When the processing was completed'))
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'timestamp',
        'weight' => 1,
      ])
      ->setDisplayOptions('form', [
        'type' => 'datetime_timestamp',
        'weight' => 1,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['error_message'] = BaseFieldDefinition::create('text_long')
      ->setLabel(t('Error Message'))
      ->setDescription(t('Error message if processing failed'))
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'text_default',
        'weight' => 2,
      ])
      ->setDisplayOptions('form', [
        'type' => 'text_textarea',
        'weight' => 2,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['status'] = BaseFieldDefinition::create('boolean')
      ->setLabel(t('Published'))
      ->setDescription(t('A boolean indicating whether the record is published.'))
      ->setDefaultValue(TRUE)
      ->setSetting('on_label', 'Published')
      ->setDisplayOptions('form', [
        'type' => 'boolean_checkbox',
        'weight' => 3,
      ])
      ->setDisplayConfigurable('form', TRUE);

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Created'))
      ->setDescription(t('The time that the record was created.'))
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'timestamp',
        'weight' => 4,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(t('Changed'))
      ->setDescription(t('The time that the record was last edited.'))
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'timestamp',
        'weight' => 5,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    return $fields;
  }

}
