<?php

declare(strict_types=1);

namespace Drupal\Tests\image_to_media_swapper\FunctionalJavascript;

use Drupal\editor\Entity\Editor;
use Drupal\file\FileInterface;
use Drupal\filter\Entity\FilterFormat;
use Drupal\FunctionalJavascriptTests\WebDriverTestBase;
use Drupal\media\MediaInterface;
use Drupal\Tests\ckeditor5\Traits\CKEditor5TestTrait;
use Drupal\Tests\media\Traits\MediaTypeCreationTrait;
use Drupal\Tests\TestFileCreationTrait;
use Drupal\file\Entity\File;
use Drupal\media\Entity\Media;
use Drupal\node\Entity\Node;
use Drupal\user\RoleInterface;
use Drupal\user\UserInterface;

/**
 * Tests CKEditor media swapping functionality.
 *
 * @group image_to_media_swapper
 * @group ckeditor5
 */
class MediaSwapperCKEditorTestDisabled extends WebDriverTestBase {

  use CKEditor5TestTrait;
  use MediaTypeCreationTrait;
  use TestFileCreationTrait;

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'node',
    'ckeditor5',
    'file',
    'image',
    'media',
    'media_library',
    'image_to_media_swapper',
    'serialization',
  ];

  /**
   * A user with permission to create content and use media.
   *
   * @var \Drupal\user\UserInterface
   */
  protected UserInterface $authorUser;

  /**
   * Test file entity.
   *
   * @var \Drupal\file\FileInterface
   */
  protected FileInterface $testFile;

  /**
   * Test media entity.
   *
   * @var \Drupal\media\MediaInterface
   */
  protected MediaInterface $testMedia;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create page content type.
    $this->drupalCreateContentType(['type' => 'page', 'name' => 'Basic page']);

    // Create image media type.
    $this->createMediaType('image', ['id' => 'image']);

    // Setup text format with media swapper.
    $this->setupTextFormatWithMediaSwapper();

    // Create user with necessary permissions.
    $this->authorUser = $this->drupalCreateUser([
      'administer filters',
      'create page content',
      'edit own page content',
      'create media',
      'access media overview',
      'access content',
      'use text format ckeditor5',
    ]);

    // Create a test image file.
    $test_files = $this->getTestFiles('image');
    $this->testFile = File::create([
      'uri' => $test_files[0]->uri,
      'filename' => $test_files[0]->filename,
      'filemime' => mime_content_type($test_files[0]->uri),
      'status' => 1,
    ]);
    $this->testFile->save();

    // Create test media entity.
    $this->testMedia = Media::create([
      'bundle' => 'image',
      'name' => 'Test Image',
      'field_media_image' => [
        'target_id' => $this->testFile->id(),
        'alt' => 'Test image alt text',
      ],
    ]);
    $this->testMedia->save();

    $this->drupalLogin($this->authorUser);
  }

  /**
   * Sets up text format with media swapper functionality.
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  protected function setupTextFormatWithMediaSwapper(): void {
    // Create filter format.
    FilterFormat::create([
      'format' => 'ckeditor5',
      'name' => 'CKEditor 5',
      'roles' => [RoleInterface::AUTHENTICATED_ID],
      'filters' => [
        'filter_html' => [
          'status' => TRUE,
          'settings' => [
            'allowed_html' => '<br> <p> <h2> <h3> <h4> <h5> <h6> <strong> <em> <img alt height src width> <drupal-media data-entity-type data-entity-uuid alt data-view-mode>',
          ],
        ],
        'filter_align' => ['status' => TRUE],
        'filter_caption' => ['status' => TRUE],
      ],
    ])->save();

    // Create editor configuration.
    Editor::create([
      'editor' => 'ckeditor5',
      'format' => 'ckeditor5',
      'settings' => [
        'toolbar' => [
          'items' => [
            'bold',
            'italic',
            'drupalInsertImage',
            'mediaSwapper',
          ],
        ],
        'plugins' => [],
      ],
      'image_upload' => [
        'status' => FALSE,
      ],
    ])->save();
  }

  /**
   * Tests that the media swapper button appears in CKEditor toolbar.
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   * @throws \Drupal\Core\Entity\EntityMalformedException|\Behat\Mink\Exception\ElementNotFoundException
   */

  /**
   * Helper method to safely click confirm button with proper waits.
   *
   * @param string $selector
   *   CSS selector for the button to click.
   */
  protected function clickConfirmButton(string $selector = '.confirm-yes'): void {
    // First wait for the confirm container to exist.
    $this->assertSession()->waitForElement('css', '.custom-confirm-container');
    // Small wait for animations.
    $this->getSession()->wait(500);

    // Try to find the requested button.
    $confirm_button = $this->getSession()->getPage()->find('css', $selector);

    // If the specific selector not found, try fallback buttons in order.
    if (!$confirm_button) {
      $possible_selectors = ['.confirm-yes', '.confirm-no', '.custom-confirm-box button'];
      foreach ($possible_selectors as $fallback_selector) {
        $confirm_button = $this->getSession()->getPage()->find('css', $fallback_selector);
        if ($confirm_button) {
          break;
        }
      }
    }

    $this->assertNotNull($confirm_button, 'No clickable button found in confirm dialog. Tried selector: ' . $selector);
    $confirm_button->click();
  }

  /**
   * Tests that the media swapper button exists in the CKEditor toolbar.
   */
  public function testMediaSwapperButtonExists(): void {
    // Create a test node.
    $node = Node::create([
      'type' => 'page',
      'title' => 'Test Node',
      'body' => [
        'value' => '<p>Test content</p>',
        'format' => 'ckeditor5',
      ],
    ]);
    $node->save();

    // Visit the node edit page.
    $this->drupalGet($node->toUrl('edit-form'));

    // Wait for CKEditor to initialize.
    $this->waitForEditor();

    // Check that the media swapper button exists in the toolbar.
    $this->assertSession()
      ->elementExists('css', '.ck-toolbar .ck-button[data-cke-tooltip-text="Convert to Media"]');
  }

  /**
   * Tests that clicking the media swapper button works.
   */
  public function testMediaSwapperButtonClick(): void {
    // Create a test node.
    $node = Node::create([
      'type' => 'page',
      'title' => 'Test Node',
      'body' => [
        'value' => '<p>Test content</p>',
        'format' => 'ckeditor5',
      ],
    ]);
    $node->save();

    // Visit the node edit page.
    $this->drupalGet($node->toUrl('edit-form'));

    // Wait for CKEditor to initialize.
    $this->waitForEditor();

    // Insert a simple image for testing.
    $file_uri = $this->testFile->getFileUri();
    $image_url = \Drupal::service('file_url_generator')
      ->generateAbsoluteString($file_uri);

    $javascript = "
      const editor = Drupal.CKEditor5Instances.get(Drupal.CKEditor5Instances.keys().next().value);
      editor.model.change(writer => {
        const imageBlock = writer.createElement('imageBlock', {
          dataEntityType: 'file',
          dataEntityUuid: '{$this->testFile->uuid()}',
          src: '{$image_url}',
          alt: 'Test image'
        });
        const root = editor.model.document.getRoot();
        writer.insert(imageBlock, root, 'end');
      });
    ";
    $this->getSession()->executeScript($javascript);

    // Wait for the image to appear.
    $this->assertSession()->waitForElement('css', '.ck-editor__editable img');

    // Click on the image to select it.
    $image = $this->getSession()
      ->getPage()
      ->find('css', '.ck-editor__editable img');
    $image->click();
    $this->getSession()->wait(500);

    // Find and click the media swapper button.
    $button = $this->getSession()
      ->getPage()
      ->find('css', '.ck-toolbar .ck-button[data-cke-tooltip-text="Convert to Media"]');
    $this->assertNotNull($button, 'Media Swapper button should be present');

    // A custom dialog should appear asking to convert the image.
    $this->assertSession()->waitForElement('css', '.custom-confirm-container');
    $this->assertSession()
      ->elementTextContains('css', '.custom-confirm-box p', 'Convert this file-based image to a media entity?');

    // Click the "Yes" button to accept the dialog.
    $this->clickConfirmButton();

    // Wait for dialog to disappear.
    $this->assertSession()
      ->waitForElementRemoved('css', '.custom-confirm-container');
  }

  /**
   * Tests that media swapper shows error for external images.
   *
   * @throws \Drupal\Core\Entity\EntityMalformedException
   * @throws \Behat\Mink\Exception\ElementNotFoundException
   * @throws \Drupal\Core\Entity\EntityStorageException|\Behat\Mink\Exception\ElementTextException
   * @throws \Behat\Mink\Exception\ExpectationException
   */
  public function testMediaSwapperWithExternalImage(): void {
    // Create a test node.
    $node = Node::create([
      'type' => 'page',
      'title' => 'Test Node',
      'body' => [
        'value' => '<p>Test content</p>',
        'format' => 'ckeditor5',
      ],
    ]);
    $node->save();

    // Visit the node edit page.
    $this->drupalGet($node->toUrl('edit-form'));

    // Wait for CKEditor to initialize.
    $this->waitForEditor();

    // Insert an external image (no file entity attributes).
    $javascript = "
      const editor = Drupal.CKEditor5Instances.get(Drupal.CKEditor5Instances.keys().next().value);
      editor.model.change(writer => {
        const imageBlock = writer.createElement('imageBlock', {
          src: 'https://example.com/external-image.jpg',
          alt: 'External image'
        });
        const root = editor.model.document.getRoot();
        writer.insert(imageBlock, root, 'end');
      });
    ";
    $this->getSession()->executeScript($javascript);

    // Wait for the image to appear and click it.
    $this->assertSession()->waitForElement('css', '.ck-editor__editable img');
    $image = $this->assertSession()->waitForElement('css', '.ck-editor__editable img');
    $this->assertNotNull($image, 'Image should be present');
    $image->click();
    $this->getSession()->wait(500);

    // Click the media swapper button - should show error.
    $button = $this->assertSession()->waitForElement('css', '.ck-toolbar .ck-button[data-cke-tooltip-text="Convert to Media"]');
    $this->assertNotNull($button, 'Media swapper button should be present');
    $button->click();
    // Wait a bit for the API call to complete and error dialog to appear.
    $this->getSession()->wait(1000);

    // Verify error dialog appears.
    $this->assertSession()->waitForElement('css', '.custom-confirm-container');
    // Get the text of the error message.
    $this->assertSession()->elementExists('css', '.custom-confirm-box p');
    $this->assertSession()
      ->elementTextContains('css', '.custom-confirm-box p', 'Convert this file-based image to a media entity?');

    // Click "Yes" to proceed with conversion.
    $this->clickConfirmButton();

    // Wait for dialog to disappear.
    $this->assertSession()
      ->waitForElementRemoved('css', '.custom-confirm-container');

    // Wait a bit for the API call to complete and error dialog to appear.
    $this->getSession()->wait(1000);

    // An error dialog should appear after the API call fails.
    $this->assertSession()->waitForElement('css', '.custom-confirm-container');
  }

  /**
   * Tests that media swapper shows error for untracked images.
   *
   * @throws \Drupal\Core\Entity\EntityMalformedException
   * @throws \Behat\Mink\Exception\ElementNotFoundException
   * @throws \Drupal\Core\Entity\EntityStorageException|\Behat\Mink\Exception\ElementTextException
   * @throws \Behat\Mink\Exception\ExpectationException
   */
  public function testMediaSwapperWithUntrackedImage(): void {
    // Create a test node.
    $node = Node::create([
      'type' => 'page',
      'title' => 'Test Node',
      'body' => [
        'value' => '<p>Test content</p>',
        'format' => 'ckeditor5',
      ],
    ]);
    $node->save();

    // Visit the node edit page.
    $this->drupalGet($node->toUrl('edit-form'));

    // Wait for CKEditor to initialize.
    $this->waitForEditor();

    // Insert a simple image for testing.
    $file_uri = $this->testFile->getFileUri();
    $image_url = \Drupal::service('file_url_generator')
      ->generateAbsoluteString($file_uri);

    $javascript = "
      const editor = Drupal.CKEditor5Instances.get(Drupal.CKEditor5Instances.keys().next().value);
      editor.model.change(writer => {
        const imageBlock = writer.createElement('imageBlock', {
          src: '{$image_url}',
          alt: 'Untracked image'
        });
        const root = editor.model.document.getRoot();
        writer.insert(imageBlock, root, 'end');
      });
    ";
    $this->getSession()->executeScript($javascript);

    // Wait for the image to appear and click it.
    $this->assertSession()->waitForElement('css', '.ck-editor__editable img');
    $image = $this->getSession()
      ->getPage()
      ->find('css', '.ck-editor__editable img');
    $image->click();
    $this->getSession()->wait(500);

    // Click the media swapper button - should show error.
    $button = $this->getSession()
      ->getPage()
      ->find('css', '.ck-toolbar .ck-button[data-cke-tooltip-text="Convert to Media"]');
    $button->click();
    // Wait a bit for the API call to complete and error dialog to appear.
    $this->getSession()->wait(1000);

    // Verify error dialog appears.
    $this->assertSession()->waitForElement('css', '.custom-confirm-container');
    // Get the text of the error message.
    $this->assertSession()->elementExists('css', '.custom-confirm-box p');
    $this->assertSession()
      ->elementTextContains('css', '.custom-confirm-box p', 'Convert this file-based image to a media entity?');

  }

  /**
   * Tests that media swapper works with valid remote files.
   *
   * @throws \Drupal\Core\Entity\EntityMalformedException
   * @throws \Drupal\Core\Entity\EntityStorageException
   * @throws \Behat\Mink\Exception\ElementNotFoundException
   */
  public function testMediaSwapperWithValidRemoteFile(): void {
    // Create a test node.
    $node = Node::create([
      'type' => 'page',
      'title' => 'Test Node',
      'body' => [
        'value' => '<p>Test content</p>',
        'format' => 'ckeditor5',
      ],
    ]);
    $node->save();

    // Visit the node edit page.
    $this->drupalGet($node->toUrl('edit-form'));

    // Wait for CKEditor to initialize.
    $this->waitForEditor();

    // Insert a remote image using a reliable placeholder service.
    $remote_image_url = 'https://via.placeholder.com/300x200.jpg';
    $javascript = "
      const editor = Drupal.CKEditor5Instances.get(Drupal.CKEditor5Instances.keys().next().value);
      editor.model.change(writer => {
        const imageBlock = writer.createElement('imageBlock', {
          src: '{$remote_image_url}',
          alt: 'Remote test image'
        });
        const root = editor.model.document.getRoot();
        writer.insert(imageBlock, root, 'end');
      });
    ";
    $this->getSession()->executeScript($javascript);

    // Wait for the image to appear and click it.
    $this->assertSession()->waitForElement('css', '.ck-editor__editable img');
    $image = $this->getSession()
      ->getPage()
      ->find('css', '.ck-editor__editable img');
    $image->click();
    $this->getSession()->wait(500);

    // Click the media swapper button.
    $button = $this->getSession()
      ->getPage()
      ->find('css', '.ck-toolbar .ck-button[data-cke-tooltip-text="Convert to Media"]');
    $this->assertNotNull($button, 'Media Swapper button should be present');
    $button->click();

    // Verify confirmation dialog appears.
    $this->assertSession()->waitForElement('css', '.custom-confirm-container');
    $this->assertSession()
      ->elementTextContains('css', '.custom-confirm-box p', 'Convert this file-based image to a media entity?');

    // Click "Yes" to proceed with conversion.
    $this->clickConfirmButton();

    // Wait for dialog to disappear.
    $this->assertSession()
      ->waitForElementRemoved('css', '.custom-confirm-container');

    // Note: This test verifies the UI interaction. The actual API call
    // to /media-api/swap-file-to-media/remote-uri would need to be mocked
    // or implemented in the test environment to test the full workflow.
  }

  /**
   * Test that clicking confirm-no leaves the content unchanged.
   *
   * @throws \Drupal\Core\Entity\EntityMalformedException
   * @throws \Drupal\Core\Entity\EntityStorageException
   * @throws \Behat\Mink\Exception\ElementNotFoundException|\Behat\Mink\Exception\ElementTextException
   */
  public function testMediaSwapperConfirmNo(): void {
    // Create a test node.
    $node = Node::create([
      'type' => 'page',
      'title' => 'Test Node',
      'body' => [
        'value' => '<p>Test content</p>',
        'format' => 'ckeditor5',
      ],
    ]);
    $node->save();

    // Visit the node edit page.
    $this->drupalGet($node->toUrl('edit-form'));

    // Wait for CKEditor to initialize.
    $this->waitForEditor();

    // Insert a simple image for testing.
    $file_uri = $this->testFile->getFileUri();
    $image_url = \Drupal::service('file_url_generator')
      ->generateAbsoluteString($file_uri);

    $javascript = "
      const editor = Drupal.CKEditor5Instances.get(Drupal.CKEditor5Instances.keys().next().value);
      editor.model.change(writer => {
        const imageBlock = writer.createElement('imageBlock', {
          dataEntityType: 'file',
          dataEntityUuid: '{$this->testFile->uuid()}',
          src: '{$image_url}',
          alt: 'Test image'
        });
        const root = editor.model.document.getRoot();
        writer.insert(imageBlock, root, 'end');
      });
    ";
    $this->getSession()->executeScript($javascript);

    // Wait for the image to appear.
    $this->assertSession()->waitForElement('css', '.ck-editor__editable img');

    // Click on the image to select it.
    $image = $this->getSession()
      ->getPage()
      ->find('css', '.ck-editor__editable img');
    $image->click();
    $this->getSession()->wait(500);

    // Find and click the media swapper button.
    $button = $this->getSession()
      ->getPage()
      ->find('css', '.ck-toolbar .ck-button[data-cke-tooltip-text="Convert to Media"]');
    $this->assertNotNull($button, 'Media Swapper button should be present');

    // Click the media swapper button.
    $button->click();

    // A custom dialog should appear asking to convert the image.
    $this->assertSession()->waitForElement('css', '.custom-confirm-container');

    // Click the "No" button to cancel conversion.
    $this->assertSession()->elementExists('css', '.custom-confirm-box p');
    $this->assertSession()
      ->elementTextContains('css', '.custom-confirm-box p', 'Convert this file-based image to a media entity?');
    $this->getSession()->getPage()->find('css', '.confirm-no')->click();
    // Wait for dialog to disappear.
    $this->assertSession()
      ->waitForElementRemoved('css', '.custom-confirm-container');
    // Verify the image is still present in the editor.
    $this->assertSession()->elementExists('css', '.ck-editor__editable img');
  }

}
