<?php

declare(strict_types=1);

namespace Drupal\image_to_media_swapper\Form;

use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Link;
use Drupal\Core\Routing\RouteProviderInterface;
use Drupal\Core\State\StateInterface;
use Drupal\image_to_media_swapper\BatchProcessorService;
use Drupal\image_to_media_swapper\MediaSwapRecordTableService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a CKEditor image to media swapper form.
 */
final class BatchSwapperForm extends FormBase {

  /**
   * The entity field manager.
   */
  protected EntityFieldManagerInterface $entityFieldManager;

  /**
   * The image to media swapper service.
   *
   * @var \Drupal\image_to_media_swapper\BatchProcessorService
   */
  protected BatchProcessorService $batchProcessorService;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected StateInterface $state;

  /**
   * The route provider service.
   *
   * @var \Drupal\Core\Routing\RouteProviderInterface
   */
  protected RouteProviderInterface $routeProvider;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * Constructs the form.
   */
  public function __construct(
    RouteProviderInterface $routeProvider,
    EntityTypeManagerInterface $entityTypeManager,
    BatchProcessorService $batchProcessorService,
    StateInterface $state,
    private readonly MediaSwapRecordTableService $tableService,
  ) {
    $this->batchProcessorService = $batchProcessorService;
    $this->entityTypeManager = $entityTypeManager;
    $this->state = $state;
    $this->routeProvider = $routeProvider;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    /** @var \Drupal\Core\Routing\RouteProviderInterface $routeProvider */
    $routeProvider = $container->get('router.route_provider');
    /** @var \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager */
    $entityTypeManager = $container->get('entity_type.manager');
    /** @var \Drupal\image_to_media_swapper\BatchProcessorService $batchProcessorService */
    $batchProcessorService = $container->get('image_to_media_swapper.batch_processor_service');
    /** @var \Drupal\Core\State\StateInterface $drupalState */
    $drupalState = $container->get('state');
    /** @var \Drupal\image_to_media_swapper\MediaSwapRecordTableService $tableService */
    $tableService = $container->get('image_to_media_swapper.table_service');
    return new self(
      $routeProvider,
      $entityTypeManager,
      $batchProcessorService,
      $drupalState,
      $tableService,
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'image_to_media_batch_swapper_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form['description'] = [
      '#markup' => $this->t('Select text fields to scan and convert images and file links to <code>&lt;drupal-media&gt;</code> elements.'),
    ];

    // Get fields with images.
    $imageFields = $this->getFieldsWithImages();
    if (!empty($imageFields)) {
      $form['image_selector'] = [
        '#type' => 'select',
        '#title' => $this->t('Text field with &lt;img&gt; tags to convert'),
        '#options' => ['' => $this->t('- None -')] + $imageFields,
        '#description' => $this->t('Select a field containing &lt;img&gt; tags to convert to media entities.'),
      ];
    }

    // Get fields with file links.
    $linkFields = $this->getFieldsWithFileLinks();
    if (!empty($linkFields)) {
      $form['link_selector'] = [
        '#type' => 'select',
        '#title' => $this->t('Text field with file links to convert'),
        '#options' => ['' => $this->t('- None -')] + $linkFields,
        '#description' => $this->t('Select a field containing &lt;a&gt; tags linking to files to convert to media entities.'),
      ];
    }

    if (empty($imageFields) && empty($linkFields)) {
      $form['no_fields'] = [
        '#markup' => $this->t('<div class="messages messages--status">No text fields with images or file links were found.</div>'),
      ];
    }
    else {
      $form['actions'] = [
        '#type' => 'actions',
        'submit' => [
          '#type' => 'submit',
          '#value' => $this->t('Scan and convert'),
          '#submit_button' => TRUE,
        ],
      ];
    }

    // Get recent processing results from MediaSwapRecord entities.
    $form['results'] = $this->tableService->buildTable('-1 day');
    $form['results']['footer'] = [
      '#markup' => Link::createFromRoute($this->t('View all records'),
        'entity.media_swap_record.collection')->toString(),
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    parent::validateForm($form, $form_state);

    $image_field = $form_state->getValue('image_selector');
    $link_field = $form_state->getValue('link_selector');

    // At least one field must be selected.
    if (empty($image_field) && empty($link_field)) {
      $form_state->setErrorByName('image_selector', $this->t('Please select at least one field to convert.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $image_field = $form_state->getValue('image_selector');
    $link_field = $form_state->getValue('link_selector');

    // Process image field if selected.
    if (!empty($image_field)) {
      $this->batchProcessorService->startBatch($image_field, 'images');
    }

    // Process link field if selected.
    if (!empty($link_field)) {
      // We'll need to update BatchProcessorService to handle link processing.
      $this->batchProcessorService->startBatch($link_field, 'links');
    }
  }

  /**
   * Gets text fields that contain image tags.
   *
   * @return array
   *   An associative array of field keys and their labels with image counts.
   */
  private function getFieldsWithImages(): array {
    $options = [];
    $allFields = $this->batchProcessorService->getEligibleTextFields();

    foreach ($allFields as $fieldKey => $fieldLabel) {
      $entities = $this->batchProcessorService->getEntitiesWithFiles($fieldKey, ['images']);
      if (!empty($entities)) {
        $count = count($entities);
        $options[$fieldKey] = $fieldLabel . ' (' . $count . ' entities with images)';
      }
    }

    return $options;
  }

  /**
   * Gets text fields that contain file links.
   *
   * @return array
   *   An associative array of field keys and their labels with link counts.
   */
  private function getFieldsWithFileLinks(): array {
    $options = [];
    $allFields = $this->batchProcessorService->getEligibleTextFields();

    foreach ($allFields as $fieldKey => $fieldLabel) {
      $entities = $this->batchProcessorService->getEntitiesWithFiles($fieldKey, ['links']);
      if (!empty($entities)) {
        $count = count($entities);
        $options[$fieldKey] = $fieldLabel . ' (' . $count . ' entities with file links)';
      }
    }

    return $options;
  }

}
