<?php

declare(strict_types=1);

namespace Drupal\image_to_media_swapper;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Link;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\image_to_media_swapper\Entity\MediaSwapRecordInterface;

/**
 * Service for building consistent MediaSwapRecord tables.
 */
final class MediaSwapRecordTableService {

  use StringTranslationTrait;

  /**
   * The logger service.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  private LoggerChannelInterface $logger;

  public function __construct(
    private readonly EntityTypeManagerInterface $entityTypeManager,
    private readonly DateFormatterInterface $dateFormatter,
    private readonly ConfigFactoryInterface $configFactory,
    private readonly LoggerChannelFactoryInterface $loggerFactory,
  ) {
    // Ensure the logger is initialized.
    $this->logger = $this->loggerFactory->get('image_to_media_swapper');
  }

  /**
   * Builds a table of MediaSwapRecord entities.
   *
   * @param string|null $from_time
   *   Optional timestamp to filter records created after this time.
   *
   * @return array
   *   Render array for the table of media swap records.
   */
  public function buildTable(?string $from_time = NULL): array {
    if ($from_time !== NULL) {
      // Get the site's default timezone.
      $time_zone = $this->configFactory->get('system.date')
        ->get('timezone.default');
      // Get the date formatter
      // Convert the provided time to the site's timezone.
      $from_time_string = $this->dateFormatter
        ->format(strtotime($from_time), 'medium', 'short', $time_zone);
    }
    else {
      $from_time_string = $this->t('All time');
    }
    try {
      $header = $this->buildHeader();
      $output['results'] = [
        '#type' => 'details',
        '#title' => $this->t('Conversion Results @time', [
          '@time' => $this->t('since @time', [
            '@time' => $from_time_string,
          ]),
        ]),
        '#open' => TRUE,
        '#attributes' => ['class' => ['file-to-media-swapper-results']],
      ];
      $swapRecordStorage = $this->entityTypeManager->getStorage('media_swap_record');
      $query = $swapRecordStorage->getQuery()
        ->accessCheck(TRUE)
        ->sort('created', 'DESC')
        ->range(0, 100);
      if ($from_time !== NULL) {
        $query->condition('created', strtotime($from_time), '>=');
      }

      $swapRecordIds = $query->execute();

      if (!empty($swapRecordIds)) {
        /** @var \Drupal\image_to_media_swapper\Entity\MediaSwapRecordInterface[] $swapRecords */
        $records = $swapRecordStorage->loadMultiple($swapRecordIds);
      }
      if (!empty($records)) {
        // Group by field selector.
        $groupedResults = [];
        /** @var \Drupal\image_to_media_swapper\Entity\MediaSwapRecordInterface $record */
        foreach ($records as $record) {
          $fieldSelector = $record->getFieldSelector();
          $groupedResults[$fieldSelector][] = $record;
        }

        if (!empty($groupedResults)) {
          foreach ($groupedResults as $fieldSelector => $records) {
            $rows = [];
            foreach ($records as $record) {
              $rows[] = $this->buildRow($record);
            }
            $output['results'][$fieldSelector] = [
              '#type' => 'details',
              '#attributes' => ['class' => ['image-to-media-swapper-result-wrapper']],
              '#title' => $this->t('Results for @field (@count records)', [
                '@field' => $fieldSelector,
                '@count' => count($records),
              ]),
            ];
            $output['results'][$fieldSelector]['table'] = [
              '#type' => 'table',
              '#header' => $header,
              '#rows' => $rows,
            ];
          }
        }
      }
    }
    catch (\Exception $e) {
      $this->logger->error($e->getMessage());
      return [
        '#markup' => $this->t('An error occurred while fetching media swap records. Please try again later.'),
      ];
    }
    return $output;
  }

  /**
   * Builds a table header for MediaSwapRecord displays.
   */
  public function buildHeader(): array {
    return [
      $this->t('Entity Type'),
      $this->t('Bundle'),
      $this->t('Entity ID'),
      $this->t('Status'),
      $this->t('Category'),
      $this->t('Processed'),
      $this->t('Link'),
    ];
  }

  /**
   * Builds a table row for a MediaSwapRecord entity.
   */
  public function buildRow(MediaSwapRecordInterface $record): array {
    $entity_type = $record->getTargetEntityType();
    $entity_id = $record->getTargetEntityId();
    $link = $this->getEntityWithRoute($entity_id, $entity_type);

    $status = $record->getStatus();
    $status_class = match ($status) {
      'completed' => 'color-success',
      'failed' => 'color-error',
      'processing' => 'color-warning',
      default => 'color-status',
    };

    $processed_time = $record->getProcessedTime();
    $processed_display = $processed_time ?
      $this->dateFormatter->format($processed_time, 'short') :
      $this->t('Not processed');

    return [
      $entity_type,
      $record->getTargetBundle(),
      $entity_id,
      [
        'data' => [
          '#markup' => '<span class="' . $status_class . '">' . ucfirst($status) . '</span>',
        ],
      ],
      $record->getBatchCategory(),
      $processed_display,
      $link ?: $this->t('N/A'),
    ];
  }

  /**
   * Gets an entity link with appropriate route.
   *
   * @throws \Drupal\Core\Entity\EntityMalformedException
   */
  private function getEntityWithRoute(int $entity_id, string $type): ?Link {
    try {
      $entity = $this->entityTypeManager->getStorage($type)->load($entity_id);
    }
    catch (\Exception $e) {
      return NULL;
    }

    if (!$entity instanceof EntityInterface) {
      return NULL;
    }

    // Try to get canonical URL.
    if ($entity->hasLinkTemplate('canonical')) {
      return $entity->toLink();
    }

    // Try edit form.
    if ($entity->hasLinkTemplate('edit-form')) {
      return $entity->toLink($this->t('Edit'), 'edit-form');
    }

    // For paragraphs, try to link to parent entity.
    if ($type === 'paragraph') {
      $parent = $entity->getParentEntity();
      if ($parent instanceof EntityInterface) {
        return $this->getEntityWithRoute($parent->id(), $parent->getEntityTypeId());
      }
    }

    return NULL;
  }

}
