<?php

namespace Drupal\Tests\image_to_media_swapper\Functional;

use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Tests\BrowserTestBase;
use Drupal\node\Entity\Node;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;

/**
 * Functional test for the BatchSwapperForm.
 *
 * @group image_to_media_swapper
 */
class BatchSwapperFormTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'node',
    'media',
    'file',
    'image',
    'filter',
    'editor',
    'field',
    'user',
    'text',
    'ckeditor5',
    'linkit',
    'image_to_media_swapper',
  ];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * Sets up the test environment.
   */
  public function setUp(): void {
    parent::setUp();
    $this->entityTypeManager = $this->container->get('entity_type.manager');

  }

  /**
   * Tests form display and functionality.
   */
  public function testBatchSwapperFormDisplaysEligibleImageFields(): void {
    $admin = $this->drupalCreateUser([
      'administer nodes',
      'administer site configuration',
      'access content',
      'administer media',
    ]);
    $this->drupalLogin($admin);

    $this->createMediaImageType();
    $this->createFilterFormatWithMediaEmbed();
    $this->createNodeTypeWithBody();

    $filesystem = \Drupal::service('file_system');

    // Ensure public:// exists and is writable.
    $directory = 'public://';
    $filesystem->prepareDirectory($directory, FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS);

    // Create the dummy file if it doesn't exist yet.
    $uri = 'public://test.jpg';
    if (!file_exists($filesystem->realpath($uri))) {
      file_put_contents($filesystem->realpath($uri), 'dummy image content');
    }

    $node = Node::create([
      'type' => 'article',
      'title' => 'Test with raw image',
      'body' => [
        'value' => '<p><img src="/sites/default/files/test.jpg" /></p>',
        'format' => 'full_html',
      ],
    ]);
    $node->save();
    // Log in as admin.
    $this->drupalGet('/admin/content/media/batch-file-to-media-swapper');
    // Check that the form has a select element for image fields.
    $this->assertSession()->fieldExists('image_selector');
    // Check that the select element has the node body field as an option.
    $this->assertSession()->optionExists('image_selector', 'node.article.body');

    $this->submitForm([
      'image_selector' => 'node.article.body',
    ], 'Scan and convert');

    // The form should now show the results after processing
    // Check that the page shows the successful processing message.
    $this->assertSession()->pageTextContains('Successfully processed');

    // Check that the results section exists.
    $this->assertSession()->elementExists('css', '.file-to-media-swapper-results');

    // Since the batch processor successfully processed 1 item, we can verify
    // the conversion worked without needing the specific node title.
    $this->assertSession()->pageTextContains('1 items');
  }

  /**
   * Tests form display and functionality.
   */
  public function testBatchSwapperFormDisplaysEligibleLinkFields(): void {
    $admin = $this->drupalCreateUser([
      'administer nodes',
      'administer site configuration',
      'access content',
      'administer media',
    ]);
    $this->drupalLogin($admin);

    $this->createMediaFileType();
    $this->createFilterFormatWithMediaEmbed();
    $this->createNodeTypeWithBody();

    $filesystem = \Drupal::service('file_system');

    // Ensure public:// exists and is writable.
    $directory = 'public://';
    $filesystem->prepareDirectory($directory, FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS);

    // Create the dummy file if it doesn't exist yet.
    $uri = 'public://test.pdf';
    if (!file_exists($filesystem->realpath($uri))) {
      file_put_contents($filesystem->realpath($uri), 'dummy pdf content');
    }

    $node = Node::create([
      'type' => 'article',
      'title' => 'Test with raw pdf link',
      'body' => [
        'value' => '<p><a href="/sites/default/files/test.pdf">Download PDF</a></p>',
        'format' => 'full_html',
      ],
    ]);
    $node->save();
    // Log in as admin.
    $this->drupalGet('/admin/content/media/batch-file-to-media-swapper');
    // Check that the form has a select element for image fields.
    $this->assertSession()->fieldExists('link_selector');
    // Check that the select element has the node body field as an option.
    $this->assertSession()->optionExists('link_selector', 'node.article.body');

    $this->submitForm([
      'link_selector' => 'node.article.body',
    ], 'Scan and convert');

    // The form should now show the results after processing
    // Check that the page shows the successful processing message.
    $this->assertSession()->pageTextContains('Successfully processed');

    // Check that the results section exists.
    $this->assertSession()->elementExists('css', '.file-to-media-swapper-results');

    // Since the batch processor successfully processed 1 item, we can verify
    // the conversion worked without needing the specific node title.
    $this->assertSession()->pageTextContains('1 items');
  }

  /**
   * Creates a media type for files.
   */
  protected function createMediaFileType(): void {
    if (!$this->entityTypeManager->getStorage('media_type')->load('file')) {
      $this->entityTypeManager->getStorage('media_type')->create([
        'id' => 'file',
        'label' => 'File',
        'source' => 'file',
        'source_configuration' => ['source_field' => 'field_media_file'],
        'field_map' => ['file' => 'field_media_file'],
      ])->save();
    }

    if (!$this->entityTypeManager->getStorage('field_storage_config')->load('media.field_media_file')) {
      FieldStorageConfig::create([
        'field_name' => 'field_media_file',
        'entity_type' => 'media',
        'type' => 'file',
      ])->save();
    }

    if (!$this->entityTypeManager->getStorage('field_config')->load('media.file.field_media_file')) {
      FieldConfig::create([
        'field_name' => 'field_media_file',
        'entity_type' => 'media',
        'bundle' => 'file',
        'label' => 'File',
        'settings' => [
          'file_extensions' => 'pdf doc docx txt',
          'file_directory' => '[date:custom:Y]-[date:custom:m]',
          'max_filesize' => '',
        ],
      ])->save();
    }
  }

  /**
   * Creates a media type for images.
   */
  protected function createMediaImageType(): void {
    if (!$this->entityTypeManager->getStorage('media_type')->load('image')) {
      $this->entityTypeManager->getStorage('media_type')->create([
        'id' => 'image',
        'label' => 'Image',
        'source' => 'image',
        'source_configuration' => ['source_field' => 'field_media_image'],
        'field_map' => ['image' => 'field_media_image'],
      ])->save();
    }

    if (!$this->entityTypeManager->getStorage('field_storage_config')->load('media.field_media_image')) {
      FieldStorageConfig::create([
        'field_name' => 'field_media_image',
        'entity_type' => 'media',
        'type' => 'image',
      ])->save();
    }

    if (!$this->entityTypeManager->getStorage('field_config')->load('media.image.field_media_image')) {
      FieldConfig::create([
        'field_name' => 'field_media_image',
        'entity_type' => 'media',
        'bundle' => 'image',
        'label' => 'Image',
      ])->save();
    }

  }

  /**
   * Creates a filter format with media embed enabled.
   */
  protected function createFilterFormatWithMediaEmbed(): void {
    if (!$this->entityTypeManager->getStorage('filter_format')->load('full_html')) {
      $this->entityTypeManager->getStorage('filter_format')->create([
        'format' => 'full_html',
        'name' => 'Full HTML',
        'filters' => [
          'media_embed' => ['status' => TRUE],
        ],
      ])->save();
    }

    if (!$this->entityTypeManager
      ->getStorage('editor')
      ->load('editor.full_html')) {
      $this->entityTypeManager
        ->getStorage('editor')->create([
          'format' => 'full_html',
          'editor' => 'ckeditor5',
        ])->save();
    }
  }

  /**
   * Creates a node type with an body field.
   */
  protected function createNodeTypeWithBody(): void {
    if (!$this->entityTypeManager->getStorage('node_type')->load('article')) {
      $this->entityTypeManager->getStorage('node_type')->create([
        'type' => 'article',
        'name' => 'Article',
      ])->save();
    }

    if (!$this->entityTypeManager->getStorage('field_config')->load('field_config.node.body')) {
      $this->entityTypeManager->getStorage('field_config')->create([
        'field_name' => 'body',
        'entity_type' => 'node',
        'bundle' => 'article',
        'label' => 'Body',
        'settings' => [
          'allowed_formats' => ['full_html'],
        ],
      ])->save();
    }

  }

}
