<?php

declare(strict_types=1);

namespace Drupal\Tests\image_to_media_swapper\Kernel;

use Drupal\Core\Entity\EntityDefinitionUpdateManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\file\FileInterface;
use Drupal\image_to_media_swapper\BatchProcessorService;
use Drupal\KernelTests\KernelTestBase;
use Drupal\node\NodeInterface;
use Drupal\user\UserInterface;

/**
 * Tests access to BatchProcessorService.
 *
 * @group image_to_media_swapper
 */
class BatchProcessorServiceKernelTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'field',
    'file',
    'media',
    'filter',
    'editor',
    'text',
    'image',
    'node',
    'options',
    'ckeditor5',
    'image_to_media_swapper',
  ];

  /**
   * The BatchProcessor service under test.
   *
   * @var \Drupal\image_to_media_swapper\BatchProcessorService
   */
  protected BatchProcessorService $service;

  /**
   * The entity definition update manager.
   *
   * @var \Drupal\Core\Entity\EntityDefinitionUpdateManagerInterface
   */
  protected EntityDefinitionUpdateManagerInterface $entityDefinitionUpdateManager;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The file entity used in tests.
   *
   * @var \Drupal\file\FileInterface
   */
  protected FileInterface $file;

  /**
   * The node entity used in tests.
   *
   * @var \Drupal\node\NodeInterface
   */
  protected NodeInterface $node;

  /**
   * The user entity used in tests.
   *
   * @var \Drupal\user\UserInterface
   */
  protected UserInterface $user;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Install required schemas.
    $this->installEntitySchema('node');
    $this->installSchema('node', ['node_access']);

    $this->installEntitySchema('user');
    $this->installEntitySchema('file');
    $this->installSchema('file', ['file_usage']);
    $this->installEntitySchema('media');

    // Install configuration for needed modules.
    $this->installConfig(['filter', 'editor', 'media']);

    // Retrieve services.
    $this->service = $this->container->get('image_to_media_swapper.batch_processor_service');
    $this->entityDefinitionUpdateManager = $this->container->get('entity.definition_update_manager');
    $this->entityTypeManager = $this->container->get('entity_type.manager');
    $fileSystem = $this->container->get('file_system');

    // Create the 'image' media type bundle if it doesn't exist.
    if (!$this->entityTypeManager->getStorage('media_type')->load('image')) {
      $this->entityTypeManager->getStorage('media_type')->create([
        'id' => 'image',
        'label' => 'Image',
        'source' => 'image',
        'status' => TRUE,
        'description' => 'Image media type',
        'source_configuration' => [
          'source_field' => 'field_media_image',
        ],
        'field_map' => [
          'image' => 'field_media_image',
        ],
      ])->save();
    }

    // Create field storage and field config for field_media_image.
    if (!$this->entityTypeManager->getStorage('field_storage_config')
      ->load('media.field_media_image')) {
      $this->entityTypeManager->getStorage('field_storage_config')->create([
        'field_name' => 'field_media_image',
        'entity_type' => 'media',
        'type' => 'image',
        'cardinality' => 1,
      ])->save();
    }

    if (!$this->entityTypeManager->getStorage('field_config')
      ->load('media.image.field_media_image')) {
      $this->entityTypeManager->getStorage('field_config')->create([
        'field_name' => 'field_media_image',
        'entity_type' => 'media',
        'bundle' => 'image',
        'label' => 'Image',
        'required' => TRUE,
      ])->save();
    }

    if (!$this->entityTypeManager->getStorage('node_type')->load('node_type.article')) {
      $this->entityTypeManager->getStorage('node_type')->create([
        'type' => 'article',
        'name' => 'Article',
      ])->save();
    }

    if (!$this->entityTypeManager->getStorage('field_storage_config')->load('field_storage_config.field_name.body')) {
      $this->entityTypeManager->getStorage('field_storage_config')->create([
        'field_name' => 'body',
        'entity_type' => 'node',
        'type' => 'text_long',
      ])->save();
    }
    if (!$this->entityTypeManager->getStorage('field_config')->load('field_config.field_name.body')) {
      $this->entityTypeManager->getStorage('field_config')->create([
        'field_name' => 'body',
        'entity_type' => 'node',
        'bundle' => 'article',
        'label' => 'Body',
        'settings' => [
          'allowed_formats' => ['full_html'],
        ],
      ])->save();
    }

    // Create a filter format with media_embed enabled.
    $format = $this->entityTypeManager->getStorage('filter_format')->create([
      'format' => 'full_html',
      'name' => 'Full HTML',
      'filters' => [
        'media_embed' => ['status' => TRUE],
      ],
    ]);
    $format->save();

    // Create an editor using CKEditor 5.
    $this->entityTypeManager->getStorage('editor')->create([
      'format' => 'full_html',
      'editor' => 'ckeditor5',
    ])->save();

    $publicDirectory = 'public://';
    // Prepare the public directory and create a dummy image file.
    $fileSystem->prepareDirectory($publicDirectory, FileSystemInterface::CREATE_DIRECTORY |
      FileSystemInterface::MODIFY_PERMISSIONS);
    /** @var \Drupal\file\FileInterface $file */
    $file = $this->entityTypeManager->getStorage('file')->create([
      'uri' => $publicDirectory . 'test.jpg',
      'status' => 1,
    ]);
    file_put_contents($file->getFileUri(), 'fake image content');
    $file->save();
    $this->file = $file;

    // Create a user.
    $user = $this->entityTypeManager->getStorage('user')->create([
      'name' => 'tester',
    ]);
    $user->save();
    $this->user = $user;

    // Create a node with a raw <img> tag in the body.
    $node = $this->entityTypeManager->getStorage('node')->create([
      'type' => 'article',
      'title' => 'Test Node',
      'body' => [
        'value' => '<p>Image: <img src="/sites/default/files/test.jpg" alt="test image" /></p>',
        'format' => 'full_html',
      ],
      'uid' => $user->id(),
    ]);
    $node->save();
    $this->node = $node;

    // Assertions to confirm the setup is valid.
    $this->assertNotNull($this->entityTypeManager->getStorage('media_type')
      ->load('image'));
    $this->assertTrue($this->entityTypeManager->getStorage('media')
      ->getEntityType()
      ->hasKey('bundle'));
    $this->assertTrue($node->hasField('body'));
  }

  /**
   * Tests that the service can find eligible text fields.
   *
   * @covers \Drupal\image_to_media_swapper\BatchProcessorService::getEligibleTextFields
   */
  public function testEligibleTextFieldsWithAllowedFormat(): void {
    $fields = $this->service->getEligibleTextFields();
    $this->assertArrayHasKey('node.article.body', $fields);
  }

  /**
   * Tests that the service can find entities with images in a specific field.
   *
   * @covers \Drupal\image_to_media_swapper\BatchProcessorService::getEntitiesWithFiles
   */
  public function testGetEntitiesWithFilesFindsNodes(): void {
    $result = $this->service->getEntitiesWithFiles('node.article.body');
    $this->assertArrayHasKey($this->node->id(), $result);
    $this->assertCount(1, $result);
  }

  /**
   * Tests that the service can find a file entity from a public URI.
   *
   * @covers \Drupal\image_to_media_swapper\BatchProcessorService::swapImagesToMedia
   */
  public function testSwapImagesToMedia(): void {
    $result = $this->service->swapImagesToMedia('node.article.body', [$this->node]);
    $this->assertArrayHasKey($this->node->id(), $result);

    $updated = $this->entityTypeManager->getStorage('node')
      ->load($this->node->id());
    $this->assertNotNull($updated, 'Updated node was loaded.');

    $value = $updated->get('body')->value;
    $this->assertStringContainsString('<drupal-media', $value);
    $this->assertStringContainsString('data-entity-uuid', $value);
  }

}
