<?php

declare(strict_types=1);

namespace Drupal\image_to_media_swapper\Entity;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityChangedInterface;
use Drupal\Core\Entity\EntityPublishedInterface;

/**
 * Provides an interface for media swap record entities.
 */
interface MediaSwapRecordInterface extends ContentEntityInterface, EntityChangedInterface, EntityPublishedInterface {

  /**
   * Gets the field selector that was processed.
   *
   * @return string
   *   The field selector in format 'entity_type.bundle.field_name'.
   */
  public function getFieldSelector(): string;

  /**
   * Sets the field selector.
   *
   * @param string $field_selector
   *   The field selector.
   *
   * @return \Drupal\image_to_media_swapper\Entity\MediaSwapRecordInterface
   *   The called media swap record entity.
   */
  public function setFieldSelector(string $field_selector): MediaSwapRecordInterface;

  /**
   * Gets the target entity type.
   *
   * @return string
   *   The target entity type.
   */
  public function getTargetEntityType(): string;

  /**
   * Sets the target entity type.
   *
   * @param string $entity_type
   *   The target entity type.
   *
   * @return \Drupal\image_to_media_swapper\Entity\MediaSwapRecordInterface
   *   The called media swap record entity.
   */
  public function setTargetEntityType(string $entity_type): MediaSwapRecordInterface;

  /**
   * Gets the target bundle.
   *
   * @return string
   *   The target bundle.
   */
  public function getTargetBundle(): string;

  /**
   * Sets the target bundle.
   *
   * @param string $bundle
   *   The target bundle.
   *
   * @return \Drupal\image_to_media_swapper\Entity\MediaSwapRecordInterface
   *   The called media swap record entity.
   */
  public function setTargetBundle(string $bundle): MediaSwapRecordInterface;

  /**
   * Gets the target entity ID.
   *
   * @return int
   *   The target entity ID.
   */
  public function getTargetEntityId(): int;

  /**
   * Gets the target entity.
   *
   * @return \Drupal\Core\Entity\EntityInterface|null
   *   The target entity or NULL if not found.
   */
  public function getTargetEntity(): ?EntityInterface;

  /**
   * Sets the target entity ID.
   *
   * @param int $entity_id
   *   The target entity ID.
   *
   * @return \Drupal\image_to_media_swapper\Entity\MediaSwapRecordInterface
   *   The called media swap record entity.
   */
  public function setTargetEntityId(int $entity_id): MediaSwapRecordInterface;

  /**
   * Gets the batch category.
   *
   * @return string
   *   The batch category ('images', 'links', etc.).
   */
  public function getBatchCategory(): string;

  /**
   * Sets the batch category.
   *
   * @param string $category
   *   The batch category.
   *
   * @return \Drupal\image_to_media_swapper\Entity\MediaSwapRecordInterface
   *   The called media swap record entity.
   */
  public function setBatchCategory(string $category): MediaSwapRecordInterface;

  /**
   * Gets the processing status.
   *
   * @return string
   *   The processing status ('completed', 'failed', 'processing', etc.).
   */
  public function getStatus(): string;

  /**
   * Sets the processing status.
   *
   * @param string $status
   *   The processing status.
   *
   * @return \Drupal\image_to_media_swapper\Entity\MediaSwapRecordInterface
   *   The called media swap record entity.
   */
  public function setStatus(string $status): MediaSwapRecordInterface;

  /**
   * Gets the processed timestamp.
   *
   * @return int
   *   The processed timestamp.
   */
  public function getProcessedTime(): int;

  /**
   * Sets the processed timestamp.
   *
   * @param int $timestamp
   *   The processed timestamp.
   *
   * @return \Drupal\image_to_media_swapper\Entity\MediaSwapRecordInterface
   *   The called media swap record entity.
   */
  public function setProcessedTime(int $timestamp): MediaSwapRecordInterface;

  /**
   * Gets the error message if processing failed.
   *
   * @return string|null
   *   The error message or NULL if no error.
   */
  public function getErrorMessage(): ?string;

  /**
   * Sets the error message.
   *
   * @param string|null $message
   *   The error message or NULL to clear.
   *
   * @return \Drupal\image_to_media_swapper\Entity\MediaSwapRecordInterface
   *   The called media swap record entity.
   */
  public function setErrorMessage(?string $message): MediaSwapRecordInterface;

  /**
   * Gets the source file ID that was converted.
   *
   * @return int|null
   *   The source file ID or NULL if not set.
   */
  public function getSourceFileId(): ?int;

  /**
   * Sets the source file ID.
   *
   * @param int|null $file_id
   *   The source file ID or NULL to clear.
   *
   * @return \Drupal\image_to_media_swapper\Entity\MediaSwapRecordInterface
   *   The called media swap record entity.
   */
  public function setSourceFileId(?int $file_id): MediaSwapRecordInterface;

  /**
   * Gets the created media ID.
   *
   * @return int|null
   *   The created media ID or NULL if not set.
   */
  public function getCreatedMediaId(): ?int;

  /**
   * Sets the created media ID.
   *
   * @param int|null $media_id
   *   The created media ID or NULL to clear.
   *
   * @return \Drupal\image_to_media_swapper\Entity\MediaSwapRecordInterface
   *   The called media swap record entity.
   */
  public function setCreatedMediaId(?int $media_id): MediaSwapRecordInterface;

  /**
   * Gets the conversion details.
   *
   * @return string|null
   *   The conversion details or NULL if not set.
   */
  public function getConversionDetails(): ?string;

  /**
   * Sets the conversion details.
   *
   * @param string|null $details
   *   The conversion details or NULL to clear.
   *
   * @return \Drupal\image_to_media_swapper\Entity\MediaSwapRecordInterface
   *   The called media swap record entity.
   */
  public function setConversionDetails(?string $details): MediaSwapRecordInterface;

}
