import {Plugin} from 'ckeditor5/src/core';
import {ButtonView} from 'ckeditor5/src/ui';

export default class LogSelectedElement extends Plugin {
  init() {
    const editor = this.editor;

    editor.ui.componentFactory.add('logSelected', locale => {
      const button = new ButtonView(locale);

      button.set({
        label: 'Log Selected',
        tooltip: true,
        withText: true
      });

      button.on('execute', () => {
        const selection = editor.model.document.selection;
        const selectedElement = selection.getSelectedElement();
        const position = selection.getFirstPosition();

        console.log('--- LOGGING SELECTION ---');

        console.log('Selected element:', selectedElement);

        // Get more details about the selected element
        if (selectedElement) {
          console.log('Selected element name:', selectedElement.name);
          console.log('Selected element attributes:', Array.from(selectedElement.getAttributes()));

          // Check for specific types of elements
          if (selectedElement.name === 'imageInline' || selectedElement.name === 'imageBlock') {
            console.log('IMAGE ELEMENT DETECTED!');
            console.log('Image src:', selectedElement.getAttribute('src'));
            console.log('Image alt:', selectedElement.getAttribute('alt'));

            // Check for data attributes that might be relevant for media conversion
            const dataEntityType = selectedElement.getAttribute('dataEntityType');
            const dataEntityUuid = selectedElement.getAttribute('dataEntityUuid');
            if (dataEntityType || dataEntityUuid) {
              console.log('DRUPAL ENTITY DATA DETECTED:');
              console.log('  data-entity-type:', dataEntityType);
              console.log('  data-entity-uuid:', dataEntityUuid);
            }
          }

          // Check for link elements
          if (selectedElement.hasAttribute('linkHref')) {
            console.log('LINK ELEMENT DETECTED!');
            console.log('Link href:', selectedElement.getAttribute('linkHref'));

            // Additional link information
            console.log('Link target:', selectedElement.getAttribute('linkTarget'));
            console.log('Link download:', selectedElement.getAttribute('linkDownload'));

            // Check if this might be a file link
            const href = selectedElement.getAttribute('linkHref');
            if (href) {
              const isFileLink = this.detectPossibleFileLink(href);
              console.log('Appears to be a file link:', isFileLink);
              if (isFileLink) {
                console.log('Possible file extension:', this.extractFileExtension(href));
              }
            }
          }

          // Check for drupalMedia elements
          if (selectedElement.name === 'drupalMedia') {
            console.log('DRUPAL MEDIA ELEMENT DETECTED!');
            console.log('Media data-entity-type:', selectedElement.getAttribute('dataEntityType'));
            console.log('Media data-entity-uuid:', selectedElement.getAttribute('dataEntityUuid'));
            console.log('Media data-view-mode:', selectedElement.getAttribute('dataViewMode'));
          }
        }

        console.log('First position parent:', position.parent.name);
        console.log('First position parent attrs:', Array.from(position.parent.getAttributes()));
        console.log('First position:', selection.getFirstPosition().parent.name);
        console.log('Last position:', selection.getLastPosition().parent.name);

        // Log the current editor model state
        console.log('Current editor model content:');
        console.log(editor.getData());

        // Log selection ranges
        const ranges = Array.from(selection.getRanges());
        console.log('Selection ranges:', ranges);

        // Log model schema information for the selected element
        if (selectedElement) {
          const schema = editor.model.schema;
          console.log('Schema information:');
          console.log('  isBlock:', schema.isBlock(selectedElement));
          console.log('  isInline:', schema.isInline(selectedElement));
          console.log('  isObject:', schema.isObject(selectedElement));
          console.log('  allowedAttributes:', schema.getAttributesAllowedOnElement(selectedElement.name));
        }

        console.log('--------------------------');
      });

      return button;
    });
  }

  /**
   * Detects if a URL appears to be a file link
   * @param {string} url - The URL to check
   * @return {boolean} - Whether the URL appears to be a file link
   */
  detectPossibleFileLink(url) {
    // Common file extensions to check for
    const fileExtensions = ['pdf', 'doc', 'docx', 'xls', 'xlsx', 'ppt', 'pptx', 'zip', 'rar', 'jpg', 'jpeg', 'png', 'gif', 'svg'];

    // Check if URL ends with a file extension
    const urlLower = url.toLowerCase();
    for (const ext of fileExtensions) {
      if (urlLower.endsWith(`.${ext}`) || urlLower.includes(`.${ext}?`)) {
        return true;
      }
    }

    // Check for download parameters in URL
    return urlLower.includes('download=') || urlLower.includes('/download/') || urlLower.includes('/files/');


  }

  /**
   * Extracts file extension from a URL
   * @param {string} url - The URL to extract from
   * @return {string|null} - The file extension or null
   */
  extractFileExtension(url) {
    const urlPath = url.split('?')[0]; // Remove query parameters
    const matches = urlPath.match(/\.([a-zA-Z0-9]+)$/);
    return matches ? matches[1].toLowerCase() : null;
  }
}
