<?php

declare(strict_types=1);

namespace Drupal\Tests\image_to_media_swapper\Functional;

use Drupal\Tests\BrowserTestBase;
use Drupal\Tests\image_to_media_swapper\Traits\MediaFieldSetupTrait;
use Drupal\user\UserInterface;

/**
 * Tests batch swapper access control functionality.
 *
 * @group image_to_media_swapper
 */
class BatchSwapperAccessTest extends BrowserTestBase {

  use MediaFieldSetupTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'image_to_media_swapper',
    'media',
    'file',
    'image',
    'options',
    'filter',
    'editor',
    'field',
    'user',
    'text',
    'ckeditor5',
    'linkit',
  // Explicitly include the node module.
    'node',
  // Field UI module may be needed for field creation.
    'field_ui',
    'system',
  ];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * Admin user with permission to use batch swapper.
   *
   * @var \Drupal\user\UserInterface
   */
  protected UserInterface $adminUser;

  /**
   * Regular user without batch swapper permissions.
   *
   * @var \Drupal\user\UserInterface
   */
  protected UserInterface $regularUser;

  /**
   * The route to the batch swapper form.
   *
   * @var string
   */
  protected string $batchSwapperRoute;

  /**
   * The route to the security settings form.
   *
   * @var string
   */
  protected string $securitySettingsRoute;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create media types for testing.
    $this->createMediaImageType();
    $this->createFilterFormatWithMediaEmbed();
    $this->createNodeTypeWithBody();

    // Set up routes.
    $routeProvider = \Drupal::service('router.route_provider');
    $this->batchSwapperRoute = $routeProvider->getRouteByName('image_to_media_swapper.batch_swapper_form')->getPath();
    $this->securitySettingsRoute = $routeProvider->getRouteByName('image_to_media_swapper.security_settings')->getPath();

    // Create admin user with both permissions.
    $this->adminUser = $this->drupalCreateUser([
      'administer site configuration',
      'access batch media swapper',
      'administer media',
      'access content',
    ]);

    // Create regular user without batch swapper permissions.
    $this->regularUser = $this->drupalCreateUser([
      'access content',
    ]);
  }

  /**
   * Tests access control based on permissions.
   */
  public function testPermissionBasedAccess(): void {
    // Ensure batch processing is enabled.
    $this->configureSecuritySettings(FALSE);

    // Explicitly log out to test anonymous access.
    $this->drupalLogout();

    // Test access is denied for anonymous users.
    $this->drupalGet($this->batchSwapperRoute);
    $this->assertSession()->statusCodeEquals(403);

    // Test access is denied for users without the permission.
    $this->drupalLogin($this->regularUser);
    $this->drupalGet($this->batchSwapperRoute);
    $this->assertSession()->statusCodeEquals(403);

    // Test access is granted for users with the permission.
    $this->drupalLogin($this->adminUser);
    $this->drupalGet($this->batchSwapperRoute);
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('File to Media Swapper Batch');
  }

  /**
   * Tests access control based on the disable batch processing setting.
   */
  public function testDisableBatchProcessingSetting(): void {
    // First test with batch processing enabled.
    $this->configureSecuritySettings(FALSE);

    // Admin user should have access.
    $this->drupalLogin($this->adminUser);
    $this->drupalGet($this->batchSwapperRoute);
    $this->assertSession()->statusCodeEquals(200);

    // Now disable batch processing.
    $this->configureSecuritySettings(TRUE);

    // Admin user should be denied access despite having permission.
    $this->drupalGet($this->batchSwapperRoute);
    $this->assertSession()->statusCodeEquals(403);

    // Re-enable batch processing.
    $this->configureSecuritySettings(FALSE);

    // Admin user should regain access.
    $this->drupalGet($this->batchSwapperRoute);
    $this->assertSession()->statusCodeEquals(200);
  }

  /**
   * Helper method to configure security settings.
   *
   * @param bool $disableBatchProcessing
   *   Whether to disable batch processing.
   */
  protected function configureSecuritySettings(bool $disableBatchProcessing): void {
    // Ensure we're logged in as admin.
    $this->drupalLogin($this->adminUser);

    // Configure the security settings.
    $edit = [
      'disable_batch_processing' => $disableBatchProcessing,
    ];
    $this->drupalGet($this->securitySettingsRoute);
    $this->submitForm($edit, 'Save configuration');
    $this->assertSession()->pageTextContains('The configuration options have been saved.');

    // Verify configuration was saved.
    $config = $this->config('image_to_media_swapper.security_settings');
    $this->assertEquals($disableBatchProcessing, $config->get('disable_batch_processing'));
  }

}
