<?php

declare(strict_types=1);

namespace Drupal\Tests\image_to_media_swapper\Kernel;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\KernelTests\KernelTestBase;

/**
 * Tests configuration installation and schema validation.
 *
 * @group image_to_media_swapper
 * @group configuration
 */
class ConfigurationTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'image_to_media_swapper',
    'system',
  ];

  /**
   * The configuration factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  private ConfigFactoryInterface $configFactory;

  /**
   * {@inheritdoc}
   */
  public function setUp(): void {
    parent::setUp();

    // Get the configuration factory service.
    $this->configFactory = $this->container->get('config.factory');
  }

  /**
   * Tests that install configuration is properly loaded.
   */
  public function testInstallConfiguration(): void {
    $this->installConfig(['image_to_media_swapper']);

    $config = $this->config('image_to_media_swapper.security_settings');

    // Test default values match install config.
    $this->assertTrue($config->get('enable_remote_downloads'));
    $this->assertEquals(10, $config->get('max_file_size'));
    $this->assertEquals(30, $config->get('download_timeout'));
    $this->assertFalse($config->get('restrict_domains'));
    $this->assertTrue($config->get('block_private_ips'));
    $this->assertFalse($config->get('require_https'));
    $this->assertEquals(3, $config->get('max_redirects'));

    // Test that arrays are properly initialized.
    $extensions = $config->get('allowed_extensions_array');
    $this->assertIsArray($extensions);
    $this->assertContains('jpg', $extensions);
    $this->assertContains('png', $extensions);
    $this->assertContains('pdf', $extensions);

    $mimeTypes = $config->get('allowed_mime_types_array');
    $this->assertIsArray($mimeTypes);
    $this->assertContains('image/jpeg', $mimeTypes);
    $this->assertContains('image/png', $mimeTypes);
    $this->assertContains('application/pdf', $mimeTypes);
  }

  /**
   * Tests configuration schema validation.
   */
  public function testConfigurationSchema(): void {
    $this->installConfig(['image_to_media_swapper']);

    // Get the typed config manager.
    $typedConfig = \Drupal::service('config.typed');

    // Load the configuration and validate against schema.
    $config = $this->config('image_to_media_swapper.security_settings');
    $definition = $typedConfig->getDefinition('image_to_media_swapper.security_settings');
    $typedConfigData = $typedConfig->createFromNameAndData('image_to_media_swapper.security_settings', $config->get());

    // Validate the configuration structure.
    $this->assertInstanceOf('Drupal\Core\Config\Schema\Mapping', $typedConfigData);

    // Test that all required properties exist and have correct types.
    $this->assertIsBool($typedConfigData->get('enable_remote_downloads')->getValue());
    $this->assertIsInt($typedConfigData->get('max_file_size')->getValue());
    $this->assertIsInt($typedConfigData->get('download_timeout')->getValue());
    $this->assertIsBool($typedConfigData->get('restrict_domains')->getValue());
    $this->assertIsString($typedConfigData->get('allowed_domains_list')->getValue());
    $this->assertIsArray($typedConfigData->get('allowed_domains')->getValue());
    $this->assertIsArray($typedConfigData->get('allowed_extensions_array')->getValue());
    $this->assertIsArray($typedConfigData->get('allowed_mime_types_array')->getValue());
    $this->assertIsBool($typedConfigData->get('block_private_ips')->getValue());
    $this->assertIsBool($typedConfigData->get('require_https')->getValue());
    $this->assertIsInt($typedConfigData->get('max_redirects')->getValue());
  }

  /**
   * Tests configuration modification and persistence.
   */
  public function testConfigurationModification(): void {
    $this->installConfig(['image_to_media_swapper']);

    $config = $this->configFactory->getEditable('image_to_media_swapper.security_settings');

    // Modify configuration values.
    $config->set('enable_remote_downloads', FALSE);
    $config->set('max_file_size', 5);
    $config->set('restrict_domains', TRUE);
    $config->set('allowed_domains', ['example.com', '*.trusted.org']);
    $config->set('allowed_extensions_array', ['jpg', 'png']);
    $config->set('allowed_mime_types_array', ['image/jpeg', 'image/png']);
    $config->set('require_https', TRUE);
    $config->save();

    // Reload configuration and verify changes.
    $reloadedConfig = $this->config('image_to_media_swapper.security_settings');
    $this->assertFalse($reloadedConfig->get('enable_remote_downloads'));
    $this->assertEquals(5, $reloadedConfig->get('max_file_size'));
    $this->assertTrue($reloadedConfig->get('restrict_domains'));
    $this->assertEquals(['example.com', '*.trusted.org'], $reloadedConfig->get('allowed_domains'));
    $this->assertEquals(['jpg', 'png'], $reloadedConfig->get('allowed_extensions_array'));
    $this->assertEquals(['image/jpeg', 'image/png'], $reloadedConfig->get('allowed_mime_types_array'));
    $this->assertTrue($reloadedConfig->get('require_https'));
  }

  /**
   * Tests configuration export and import.
   */
  public function testConfigurationExportImport(): void {
    $this->installConfig(['image_to_media_swapper']);

    // Get the configuration data.
    $config = $this->config('image_to_media_swapper.security_settings');
    $originalData = $config->get();

    // Export configuration.
    $exported = \Drupal::service('config.storage')->read('image_to_media_swapper.security_settings');
    $this->assertEquals($originalData, $exported);

    // Modify and re-import.
    $exported['max_file_size'] = 20;
    $exported['require_https'] = TRUE;

    \Drupal::service('config.storage')->write('image_to_media_swapper.security_settings', $exported);

    // Clear config cache and reload.
    \Drupal::service('config.factory')->reset('image_to_media_swapper.security_settings');
    $reloadedConfig = $this->config('image_to_media_swapper.security_settings');

    $this->assertEquals(20, $reloadedConfig->get('max_file_size'));
    $this->assertTrue($reloadedConfig->get('require_https'));
  }

  /**
   * Tests configuration validation with invalid values.
   */
  public function testConfigurationValidation(): void {
    $this->installConfig(['image_to_media_swapper']);

    $config = $this->configFactory->getEditable('image_to_media_swapper.security_settings');

    // Test that configuration accepts valid boundary values.
    $config->set('max_file_size', 1);
    $config->set('download_timeout', 5);
    $config->set('max_redirects', 0);
    $config->save();

    $this->assertEquals(1, $config->get('max_file_size'));
    $this->assertEquals(5, $config->get('download_timeout'));
    $this->assertEquals(0, $config->get('max_redirects'));

    // Test maximum boundary values.
    $config->set('max_file_size', 100);
    $config->set('download_timeout', 300);
    $config->set('max_redirects', 10);
    $config->save();

    $this->assertEquals(100, $config->get('max_file_size'));
    $this->assertEquals(300, $config->get('download_timeout'));
    $this->assertEquals(10, $config->get('max_redirects'));
  }

  /**
   * Tests that configuration dependencies are correctly set.
   */
  public function testConfigurationDependencies(): void {
    $this->installConfig(['image_to_media_swapper']);

    $config = $this->config('image_to_media_swapper.security_settings');
    $dependencies = $config->get('dependencies');

    // The configuration should depend on the module.
    $this->assertArrayHasKey('enforced', $dependencies);
    $this->assertArrayHasKey('module', $dependencies['enforced']);
    $this->assertContains('image_to_media_swapper', $dependencies['enforced']['module']);
  }

}
