<?php

declare(strict_types=1);

namespace Drupal\Tests\image_to_media_swapper\Kernel;

use Symfony\Component\Routing\Route;
use Drupal\KernelTests\KernelTestBase;
use Drupal\user\Entity\Role;
use Drupal\user\Entity\User;

/**
 * Tests access to MediaSwapperController routes.
 *
 * @group image_to_media_swapper
 */
class MediaSwapperControllerAccessTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'file',
    'image',
    'media',
    'options',
    'serialization',
    'image_to_media_swapper',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->installEntitySchema('user');
    $this->installEntitySchema('file');
    $this->installEntitySchema('media');
    $this->installSchema('system', ['sequences']);
    $this->installSchema('file', ['file_usage']);

    // Rebuild routes to ensure they're available.
    \Drupal::service('router.builder')->rebuild();
  }

  /**
   * Tests access with required media permissions.
   */
  public function testAccessWithPermission(): void {
    // Create a role with required media permissions.
    $role = Role::create([
      'id' => 'test_role',
      'label' => 'Test Role',
    ]);
    $role->grantPermission('create media');
    $role->grantPermission('update media');
    $role->save();

    // Create a user with the role.
    $user = User::create([
      'name' => 'test_user',
      'uid' => 2,
    ]);
    $user->addRole('test_role');
    $user->save();

    // Test access using the access checker service directly.
    $access_checker = \Drupal::service('access_check.permission');
    $route = new Route('/media-api/swap-file-to-media/file-uuid');
    $route->setRequirement('_permission', 'create media+update media');

    $access_result = $access_checker->access($route, $user);
    $this->assertTrue($access_result->isAllowed(), 'User with media permissions should have access to the route.');
  }

  /**
   * Tests access without required permission.
   */
  public function testAccessWithoutPermission(): void {
    // Create a user without media permissions.
    $user = User::create([
      'name' => 'test_user_no_perm',
      'uid' => 3,
    ]);
    $user->save();

    // Test access using the access checker service directly.
    $access_checker = \Drupal::service('access_check.permission');
    $route = new Route('/media-api/swap-file-to-media/file-uuid');
    $route->setRequirement('_permission', 'create media+update media');

    $access_result = $access_checker->access($route, $user);
    $this->assertFalse($access_result->isAllowed(), 'User without media permissions should not have access to the route.');

    // Repeat with the file path route.
    $route = new Route('/media-api/swap-file-to-media/local-path');
    $route->setRequirement('_permission', 'create media+update media');
    $access_result = $access_checker->access($route, $user);
    $this->assertFalse($access_result->isAllowed(), 'User without media permissions should not have access to the file path route.');
  }

  /**
   * Tests anonymous user access.
   */
  public function testAnonymousUserAccess(): void {
    // Anonymous user (uid 0).
    $anonymous_user = User::getAnonymousUser();

    // Test access using the access checker service directly.
    $access_checker = \Drupal::service('access_check.permission');
    $route = new Route('/media-api/swap-file-to-media/file-uuid');
    $route->setRequirement('_permission', 'create media+update media');

    $access_result = $access_checker->access($route, $anonymous_user);
    $this->assertFalse($access_result->isAllowed(), 'Anonymous user should not have access to the route without permission.');

    // Repeat with the file path route.
    $route = new Route('/media-api/swap-file-to-media/local-path');
    $route->setRequirement('_permission', 'create media+update media');
    $access_result = $access_checker->access($route, $anonymous_user);
    $this->assertFalse($access_result->isAllowed(), 'Anonymous user should not have access to the file path route without permission.');
  }

}
