<?php

declare(strict_types=1);

namespace Drupal\image_to_media_swapper\Access;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Routing\Access\AccessInterface;
use Drupal\Core\Session\AccountInterface;

/**
 * Checks access for batch processing functionality.
 */
class BatchSwapperAccessChecker implements AccessInterface {

  /**
   * The config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs a BatchSwapperAccessChecker object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory service.
   */
  public function __construct(ConfigFactoryInterface $config_factory) {
    $this->configFactory = $config_factory;
  }

  /**
   * Checks access to batch swapper functionality.
   *
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The currently logged in account.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  public function access(AccountInterface $account): AccessResultInterface {
    // Check if batch processing is globally disabled in security settings.
    $config = $this->configFactory->get('image_to_media_swapper.security_settings');
    $batchProcessingDisabled = $config->get('disable_batch_processing') ?? FALSE;

    if ($batchProcessingDisabled) {
      return AccessResult::forbidden('Batch processing has been disabled by site administrator.')
        ->addCacheableDependency($config);
    }

    // If the user is anonymous, always deny access.
    if ($account->isAnonymous()) {
      return AccessResult::forbidden('Anonymous users cannot access batch processing.');
    }

    // If batch processing is enabled, check if the user has the permission.
    return AccessResult::allowedIfHasPermission($account, 'access batch media swapper')
      ->addCacheableDependency($config);
  }

}
