<?php

declare(strict_types=1);

namespace Drupal\image_to_media_swapper\Form;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Link;
use Drupal\Core\Routing\RouteProviderInterface;
use Drupal\Core\State\StateInterface;
use Drupal\image_to_media_swapper\BatchProcessorService;
use Drupal\image_to_media_swapper\MediaSwapFormService;
use Drupal\image_to_media_swapper\MediaSwapRecordTableService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a CKEditor image to media swapper form.
 */
final class BatchSwapperForm extends FormBase {

  /**
   * The image to media swapper service.
   *
   * @var \Drupal\image_to_media_swapper\BatchProcessorService
   */
  protected BatchProcessorService $batchProcessorService;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected StateInterface $state;

  /**
   * The route provider service.
   *
   * @var \Drupal\Core\Routing\RouteProviderInterface
   */
  protected RouteProviderInterface $routeProvider;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The media swap form service.
   *
   * @var \Drupal\image_to_media_swapper\MediaSwapFormService
   */
  protected MediaSwapFormService $mediaSwapFormService;

  /**
   * Constructs the form.
   */
  public function __construct(
    RouteProviderInterface $routeProvider,
    EntityTypeManagerInterface $entityTypeManager,
    BatchProcessorService $batchProcessorService,
    StateInterface $state,
    private readonly MediaSwapRecordTableService $tableService,
    MediaSwapFormService $mediaSwapFormService,
  ) {
    $this->batchProcessorService = $batchProcessorService;
    $this->entityTypeManager = $entityTypeManager;
    $this->state = $state;
    $this->routeProvider = $routeProvider;
    $this->mediaSwapFormService = $mediaSwapFormService;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    /** @var \Drupal\Core\Routing\RouteProviderInterface $routeProvider */
    $routeProvider = $container->get('router.route_provider');
    /** @var \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager */
    $entityTypeManager = $container->get('entity_type.manager');
    /** @var \Drupal\image_to_media_swapper\BatchProcessorService $batchProcessorService */
    $batchProcessorService = $container->get('image_to_media_swapper.batch_processor_service');
    /** @var \Drupal\Core\State\StateInterface $drupalState */
    $drupalState = $container->get('state');
    /** @var \Drupal\image_to_media_swapper\MediaSwapRecordTableService $tableService */
    $tableService = $container->get('image_to_media_swapper.table_service');
    /** @var \Drupal\image_to_media_swapper\MediaSwapFormService $mediaSwapFormService */
    $mediaSwapFormService = $container->get('image_to_media_swapper.form_service');
    return new self(
      $routeProvider,
      $entityTypeManager,
      $batchProcessorService,
      $drupalState,
      $tableService,
      $mediaSwapFormService,
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'image_to_media_batch_swapper_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form['description'] = [
      '#markup' => $this->t('Select text fields to scan and convert images and file links to <code>&lt;drupal-media&gt;</code> elements.'),
    ];

    // Get fields with images.
    $imageFields = $this->mediaSwapFormService->getFieldsWithImages();
    if (!empty($imageFields)) {
      $form['image_selector'] = [
        '#type' => 'select',
        '#title' => $this->t('Text field with &lt;img&gt; tags to convert'),
        '#options' => ['' => $this->t('- None -')] + $imageFields,
        '#description' => $this->t('Select a field containing &lt;img&gt; tags to convert to media entities.'),
      ];
    }

    // Get fields with file links.
    $linkFields = $this->mediaSwapFormService->getFieldsWithFileLinks();
    if (!empty($linkFields)) {
      $form['link_selector'] = [
        '#type' => 'select',
        '#title' => $this->t('Text field with file links to convert'),
        '#options' => ['' => $this->t('- None -')] + $linkFields,
        '#description' => $this->t('Select a field containing &lt;a&gt; tags linking to files to convert to media entities.'),
      ];
    }

    if (empty($imageFields) && empty($linkFields)) {
      $form['no_fields'] = [
        '#markup' => $this->t('<div class="messages messages--status">No text fields with images or file links were found.</div>'),
      ];
    }
    else {
      $form['actions'] = [
        '#type' => 'actions',
        'submit' => [
          '#type' => 'submit',
          '#value' => $this->t('Scan and convert'),
          '#submit_button' => TRUE,
        ],
      ];
    }

    // Get recent processing results from MediaSwapRecord entities.
    $form['results'] = $this->tableService->buildTable('-1 day', 50);
    $form['results']['footer'] = [
      '#markup' => Link::createFromRoute($this->t('View all records'),
        'entity.media_swap_record.collection')->toString(),
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    parent::validateForm($form, $form_state);

    $image_field = $form_state->getValue('image_selector');
    $link_field = $form_state->getValue('link_selector');

    // At least one field must be selected.
    if (empty($image_field) && empty($link_field)) {
      $form_state->setErrorByName('image_selector', $this->t('Please select at least one field to convert.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $image_field = $form_state->getValue('image_selector');
    $link_field = $form_state->getValue('link_selector');

    // Process image field if selected.
    if (!empty($image_field)) {
      $this->batchProcessorService->startBatch($image_field, 'images');
    }

    // Process link field if selected.
    if (!empty($link_field)) {
      $this->batchProcessorService->startBatch($link_field, 'links');
    }
  }

}
