<?php

namespace Drupal\Tests\image_to_media_swapper\Functional;

use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Tests\BrowserTestBase;
use Drupal\Tests\image_to_media_swapper\Traits\MediaFieldSetupTrait;
use Drupal\node\Entity\Node;

/**
 * Functional test for the BatchSwapperForm.
 *
 * @group image_to_media_swapper
 */
class BatchSwapperFormTest extends BrowserTestBase {

  use MediaFieldSetupTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'node',
    'media',
    'file',
    'image',
    'options',
    'filter',
    'editor',
    'field',
    'user',
    'text',
    'ckeditor5',
    'linkit',
    'image_to_media_swapper',
  ];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * Sets up the test environment.
   */
  public function setUp(): void {
    parent::setUp();
    $this->entityTypeManager = $this->container->get('entity_type.manager');

    // Enable batch processing for testing by updating the security settings.
    $config = $this->config('image_to_media_swapper.security_settings');
    $config->set('disable_batch_processing', FALSE)->save();
  }

  /**
   * Tests form display and functionality.
   */
  public function testBatchSwapperFormDisplaysEligibleImageFields(): void {
    $admin = $this->drupalCreateUser([
      'administer nodes',
      'administer site configuration',
      'access content',
      'administer media',
      'access batch media swapper',
    ]);
    $this->drupalLogin($admin);

    $this->createMediaImageType();
    $this->createFilterFormatWithMediaEmbed();
    $this->createNodeTypeWithBody();

    $filesystem = \Drupal::service('file_system');

    // Ensure public:// exists and is writable.
    $directory = 'public://';
    $filesystem->prepareDirectory($directory, FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS);

    // Create the dummy file if it doesn't exist yet.
    $uri = 'public://test.jpg';
    if (!file_exists($filesystem->realpath($uri))) {
      file_put_contents($filesystem->realpath($uri), 'dummy image content');
    }

    $node = Node::create([
      'type' => 'article',
      'title' => 'Test with raw image',
      'body' => [
        'value' => '<p><img src="/sites/default/files/test.jpg" /></p>',
        'format' => 'full_html',
      ],
    ]);
    $node->save();
    // Log in as admin.
    $this->drupalGet('/admin/content/media/batch-file-to-media-swapper');

    // Debug: Check if we get the page at all or if access is denied.
    $this->assertSession()->statusCodeEquals(200);

    // Check that the form has a select element for image fields.
    $this->assertSession()->fieldExists('image_selector');
    // Check that the select element has the node body field as an option.
    $this->assertSession()->optionExists('image_selector', 'node.article.body');

    $this->submitForm([
      'image_selector' => 'node.article.body',
    ], 'Scan and convert');

    // The form should now show the results after processing
    // Check that the page shows the successful processing message.
    $this->assertSession()->pageTextContains('Successfully processed');

    // Check that the results section exists.
    $this->assertSession()->elementExists('css', '.file-to-media-swapper-results');

    // Since the batch processor successfully processed 1 item, we can verify
    // the conversion worked without needing the specific node title.
    $this->assertSession()->pageTextContains('1 items');
  }

  /**
   * Tests form display and functionality.
   */
  public function testBatchSwapperFormDisplaysEligibleLinkFields(): void {
    $admin = $this->drupalCreateUser([
      'administer nodes',
      'administer site configuration',
      'access content',
      'administer media',
      'access batch media swapper',
    ]);
    $this->drupalLogin($admin);

    $this->createMediaFileType();
    $this->createFilterFormatWithMediaEmbed();
    $this->createNodeTypeWithBody();

    $filesystem = \Drupal::service('file_system');

    // Ensure public:// exists and is writable.
    $directory = 'public://';
    $filesystem->prepareDirectory($directory, FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS);

    // Create the dummy file if it doesn't exist yet.
    $uri = 'public://test.pdf';
    if (!file_exists($filesystem->realpath($uri))) {
      file_put_contents($filesystem->realpath($uri), 'dummy pdf content');
    }

    $node = Node::create([
      'type' => 'article',
      'title' => 'Test with raw pdf link',
      'body' => [
        'value' => '<p><a href="/sites/default/files/test.pdf">Download PDF</a></p>',
        'format' => 'full_html',
      ],
    ]);
    $node->save();
    // Log in as admin.
    $this->drupalGet('/admin/content/media/batch-file-to-media-swapper');

    // Debug: Check if we get the page at all or if access is denied.
    $this->assertSession()->statusCodeEquals(200);

    // Check that the form has a select element for image fields.
    $this->assertSession()->fieldExists('link_selector');
    // Check that the select element has the node body field as an option.
    $this->assertSession()->optionExists('link_selector', 'node.article.body');

    $this->submitForm([
      'link_selector' => 'node.article.body',
    ], 'Scan and convert');

    // The form should now show the results after processing
    // Check that the page shows the successful processing message.
    $this->assertSession()->pageTextContains('Successfully processed');

    // Check that the results section exists.
    $this->assertSession()->elementExists('css', '.file-to-media-swapper-results');

    // Since the batch processor successfully processed 1 item, we can verify
    // the conversion worked without needing the specific node title.
    $this->assertSession()->pageTextContains('1 items');
  }

}
