<?php

declare(strict_types=1);

namespace Drupal\Tests\image_to_media_swapper\Traits;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;

/**
 * Provides common media field setup functionality for tests.
 */
trait MediaFieldSetupTrait {

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * Creates a media type for images with default settings.
   */
  protected function createMediaImageType(): void {
    $this->entityTypeManager = $this->entityTypeManager ?? $this->container->get('entity_type.manager');

    if (!$this->entityTypeManager->getStorage('media_type')->load('image')) {
      $this->entityTypeManager->getStorage('media_type')->create([
        'id' => 'image',
        'label' => 'Image',
        'source' => 'image',
        'source_configuration' => ['source_field' => 'field_media_image'],
        'field_map' => ['image' => 'field_media_image'],
      ])->save();
    }

    if (!$this->entityTypeManager->getStorage('field_storage_config')->load('media.field_media_image')) {
      FieldStorageConfig::create([
        'field_name' => 'field_media_image',
        'entity_type' => 'media',
        'type' => 'image',
      ])->save();
    }

    if (!$this->entityTypeManager->getStorage('field_config')->load('media.image.field_media_image')) {
      FieldConfig::create([
        'field_name' => 'field_media_image',
        'entity_type' => 'media',
        'bundle' => 'image',
        'label' => 'Image',
        'settings' => [
          'file_extensions' => 'jpg jpeg png gif webp',
          'file_directory' => '[date:custom:Y]-[date:custom:m]',
          'max_filesize' => '10 MB',
          'alt_field' => TRUE,
          'alt_field_required' => FALSE,
          'title_field' => TRUE,
          'title_field_required' => FALSE,
        ],
      ])->save();
    }
  }

  /**
   * Creates a media type for files with default settings.
   */
  protected function createMediaFileType(): void {
    $this->entityTypeManager = $this->entityTypeManager ?? $this->container->get('entity_type.manager');

    if (!$this->entityTypeManager->getStorage('media_type')->load('file')) {
      $this->entityTypeManager->getStorage('media_type')->create([
        'id' => 'file',
        'label' => 'File',
        'source' => 'file',
        'source_configuration' => ['source_field' => 'field_media_file'],
        'field_map' => ['file' => 'field_media_file'],
      ])->save();
    }

    if (!$this->entityTypeManager->getStorage('field_storage_config')->load('media.field_media_file')) {
      FieldStorageConfig::create([
        'field_name' => 'field_media_file',
        'entity_type' => 'media',
        'type' => 'file',
      ])->save();
    }

    if (!$this->entityTypeManager->getStorage('field_config')->load('media.file.field_media_file')) {
      FieldConfig::create([
        'field_name' => 'field_media_file',
        'entity_type' => 'media',
        'bundle' => 'file',
        'label' => 'File',
        'settings' => [
          'file_extensions' => 'pdf mp4 webm mp3 wav',
          'file_directory' => '[date:custom:Y]-[date:custom:m]',
          'max_filesize' => '100 MB',
        ],
      ])->save();
    }
  }

  /**
   * Creates a filter format with media embed enabled.
   */
  protected function createFilterFormatWithMediaEmbed(): void {
    $this->entityTypeManager = $this->entityTypeManager ?? $this->container->get('entity_type.manager');

    if (!$this->entityTypeManager->getStorage('filter_format')->load('full_html')) {
      $this->entityTypeManager->getStorage('filter_format')->create([
        'format' => 'full_html',
        'name' => 'Full HTML',
        'filters' => [
          'media_embed' => ['status' => TRUE],
        ],
      ])->save();
    }

    if (!$this->entityTypeManager->getStorage('editor')->load('full_html')) {
      $this->entityTypeManager->getStorage('editor')->create([
        'format' => 'full_html',
        'editor' => 'ckeditor5',
      ])->save();
    }
  }

  /**
   * Creates a node type with body field.
   */
  protected function createNodeTypeWithBody(): void {
    $this->entityTypeManager = $this->entityTypeManager ?? $this->container->get('entity_type.manager');

    if (!$this->entityTypeManager->getStorage('node_type')->load('article')) {
      $this->entityTypeManager->getStorage('node_type')->create([
        'type' => 'article',
        'name' => 'Article',
      ])->save();
    }

    if (!$this->entityTypeManager->getStorage('field_config')->load('node.article.body')) {
      FieldConfig::create([
        'field_name' => 'body',
        'entity_type' => 'node',
        'bundle' => 'article',
        'label' => 'Body',
        'settings' => [
          'allowed_formats' => ['full_html'],
        ],
      ])->save();
    }
  }

  /**
   * Creates a basic media bundle for testing.
   */
  protected function createMediaBundle(): void {
    $this->entityTypeManager = $this->entityTypeManager ?? $this->container->get('entity_type.manager');

    if (!$this->entityTypeManager->getStorage('media_type')->load('test_image')) {
      $bundle = $this->entityTypeManager->getStorage('media_type')->create([
        'id' => 'test_image',
        'label' => 'Test Image',
        'source' => 'image',
      ]);
      $bundle->save();

      // Create the source field.
      $source_field = $bundle->getSource()->createSourceField($bundle);
      $source_field->getFieldStorageDefinition()->save();
      $source_field->save();
      $bundle->set('source_configuration', [
        'source_field' => $source_field->getName(),
      ]);
      $bundle->save();
    }
  }

}
