<?php

namespace Drupal\imageapi_optimize_thumbor\Plugin\ImageAPIOptimizeProcessor;

use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Image\ImageFactory;
use Drupal\imageapi_optimize\ConfigurableImageAPIOptimizeProcessorBase;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\RequestException;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Component\Utility\Crypt;

/**
 * Uses the Thumbor service to generate optimized image URLs.
 *
 * @ImageAPIOptimizeProcessor(
 * id = "thumbor",
 * label = @Translation("Thumbor"),
 * description = @Translation("Provides Thumbor URLs for optimized images without saving files.")
 * )
 */
final class Thumbor extends ConfigurableImageAPIOptimizeProcessorBase {

  /**
   * The HTTP client to fetch the feed data with.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected $httpClient;

  /**
   * {@inheritdoc}
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, LoggerInterface $logger, ImageFactory $image_factory, ClientInterface $http_client) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $logger, $image_factory);

    $this->httpClient = $http_client;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('logger.factory')->get('imageapi_optimize'),
      $container->get('image.factory'),
      $container->get('http_client')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function applyToImage($image_uri) {
    $thumbor_url = $this->configuration['thumbor_url'] ?? '';
    $security_key = $this->configuration['security_key'] ?? '';

    if (empty($thumbor_url)) {
      $this->logger->warning('Thumbor URL is not configured.');
      return FALSE;
    }

    // This processor doesn't actually modify the image file on disk
    // It's used to indicate that Thumbor processing should be used
    // when generating the URL for this image

    // We return TRUE to indicate the processor "succeeded"
    // even though no actual file modification occurred
    return TRUE;
  }

  /**
   * Get the Thumbor URL for the given image URI without saving the processed image.
   *
   * @param string $image_uri
   * The URI of the image to process.
   * @param string $operations_string
   * (Optional) A string of Thumbor operations (e.g., "300x200", "fit-in/400x0").
   *
   * @return string|false
   * The Thumbor URL or false on failure.
   */
  public function getThumborUrl($image_uri, $operations_string = '') {
    $this->logger->debug('Thumbor Processor: getThumborUrl called for URI: @uri', ['@uri' => $image_uri]);

    $thumbor_url = $this->configuration['thumbor_url'] ?? '';
    $security_key = $this->configuration['security_key'] ?? '';

    $this->logger->debug('Thumbor Processor: URL: @url, Security key present: @key', [
      '@url' => $thumbor_url,
      '@key' => empty($security_key) ? 'NO' : 'YES'
    ]);

    if (empty($thumbor_url)) {
      $this->logger->warning('Thumbor URL is not configured.');
      return FALSE;
    }

    // Get the image file path
    $image_path = \Drupal::service('file_system')->realpath($image_uri);

    if (!$image_path || !file_exists($image_path)) {
      $this->logger->error('Could not find image file: @uri', ['@uri' => $image_uri]);
      return FALSE;
    }

   try {
      // CORREZIONE: Usa il percorso relativo alla WEB ROOT di Drupal
      
      // 1. Ottieni il percorso reale della directory pubblica (es: /var/www/html/dev-la7/shared/files)
      $public_files_path = \Drupal::service('file_system')->realpath('public://');

      // 2. Ottieni il percorso web della directory pubblica (es: sites/default/files)
      
      $public_web_path = \Drupal::service('stream_wrapper_manager')->getViaScheme('public')->getDirectoryPath();

      // 3. Rimuovi il percorso reale pubblico dal percorso reale dell'immagine
      // $image_path è (es: /var/www/html/dev-la7/shared/files/property/...)
      if (strpos($image_path, $public_files_path) === 0) {
        // Questo ci dà il percorso relativo alla cartella pubblica (es: property/tuttiiprogrammi/...)
        $relative_to_public_dir = ltrim(substr($image_path, strlen($public_files_path)), '/');
        
        // 4. Ricostruisci il percorso web completo che Thumbor si aspetta
        // (es: sites/default/files/property/tuttiiprogrammi/...)
        $relative_image_path = $public_web_path . '/' . $relative_to_public_dir;
      } else {
        $this->logger->error('Could not create relative path. Image path @img not in public files path @public', [
          '@img' => $image_path,
          '@public' => $public_files_path
        ]);
        return FALSE;
      }

      // Non codificare l'URL, ma codifica le parti del percorso (Thumbor lo vuole così)
      $encoded_image_url = implode('/', array_map('rawurlencode', explode('/', $relative_image_path)));

      // Log per debug
      $this->logger->debug('Thumbor Processor: Relative path generated: @path', ['@path' => $encoded_image_url]);

      // Construct the Thumbor URL for the image
      $thumbor_image_url = rtrim($thumbor_url, '/') . '/';

      // Costruisci il percorso dell'immagine con le operazioni
      $url_parts = [];
      if (!empty($operations_string)) {
        $url_parts[] = $operations_string;
      }
      $url_parts[] = $encoded_image_url;
      
      $image_url_with_ops = implode('/', $url_parts);

      // If security key is provided, generate signed URL
      if (!empty($security_key)) {
        // CORREZIONE: Il percorso da firmare ora include le operazioni
        $image_path_to_sign = $image_url_with_ops;

        // Create the signature using the security key
        $signature = $this->generateThumborSignature($image_path_to_sign, $security_key);

        // CORREZIONE: L'URL firmato non usa 'unsafe' e include le operazioni
        $thumbor_image_url .= $signature . '/' . $image_path_to_sign;
      } else {
        // Usa 'unsafe' solo se NON c'è una chiave (e include le operazioni)
        $thumbor_image_url .= 'unsafe/' . $image_url_with_ops;
      }

      return $thumbor_image_url;

    } catch (\Exception $e) {
      $this->logger->error('Failed to generate Thumbor URL: "%error".', [
        '%error' => $e->getMessage()
      ]);
    }

    return FALSE;
  }

  /**
   * Convert a file path to a web-accessible URL.
   *
   * @param string $file_path
   * The absolute file path.
   *
   * @return string|false
   * The web-accessible URL or false on failure.
   */
  protected function convertFilePathToUrl($file_path) {
    // Get the base URL from Drupal
    $base_url = \Drupal::request()->getSchemeAndHttpHost();

    // Get the file public path
    // CORREZIONE: Rimuovi la chiamata alla funzione defunta
    $file_public_path = \Drupal::config('system.file')->get('default_scheme') . '://';

    // Resolve the public path
    $public_path = \Drupal::service('file_system')->realpath($file_public_path);

    if ($public_path && strpos($file_path, $public_path) === 0) {
      // This is a public file, construct the web URL
      $relative_path = substr($file_path, strlen($public_path));
      return $base_url . '/sites/default/files' . $relative_path;
    }

    // For a more robust implementation, you might want to check other file schemes
    // or potentially upload the file to a location Thumbor can access

    // As a fallback, return false
    return FALSE;
  }

/**
   * Generate Thumbor URL signature.
   *
   * @param string $path
   * The image path.
   * @param string $security_key
   * The security key.
   *
   * @return string
   * The generated signature.
   */
  protected function generateThumborSignature($path, $security_key) {
    // CORREZIONE: Usa l'algoritmo di firma corretto per Thumbor (Base64 URL-safe)
    
    // 1. Calcola l'hash HMAC-SHA1, impostando 'true' per l'output binario grezzo
    $hmac = hash_hmac('sha1', $path, $security_key, TRUE);
    
    // 2. Codifica il risultato binario in Base64
    $signature = base64_encode($hmac);
    
    // 3. Rendi il risultato sicuro per gli URL
    // (sostituisci '+' con '-' e '/' con '_')
    $signature = str_replace(['+', '/'], ['-', '_'], $signature);
    
    return $signature;
  }
  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'thumbor_url' => '',
      'security_key' => '',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form['thumbor_url'] = [
      '#type' => 'url',
      '#title' => $this->t('Thumbor Server URL'),
      '#description' => $this->t('Enter the URL to your Thumbor server (e.g., http://your-thumbor-server.com).'),
      '#default_value' => $this->configuration['thumbor_url'],
      '#required' => TRUE,
    ];

    $form['security_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Security Key'),
      '#description' => $this->t('Enter the security key for your Thumbor server (if enabled).'),
      '#default_value' => $this->configuration['security_key'],
      '#required' => FALSE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::submitConfigurationForm($form, $form_state);

    $this->configuration['thumbor_url'] = $form_state->getValue('thumbor_url');
    $this->configuration['security_key'] = $form_state->getValue('security_key');
  }

}