<?php

namespace Drupal\imageapi_optimize_thumbor\Plugin\ImageEffect;

use Drupal\image\ConfigurableImageEffectBase;
use Drupal\image\ImageEffectInterface;
use Drupal\Core\Image\ImageInterface;

/**
 * Provides a Thumbor image effect.
 *
 * @ImageEffect(
 *   id = "imageapi_optimize_thumbor_effect",
 *   label = @Translation("Thumbor processing"),
 *   description = @Translation("Process image through Thumbor service"),
 *   default_configuration = {
 *     "thumbor_url" = "",
 *     "security_key" = ""
 *   }
 * )
 */
class ThumborImageEffect extends ConfigurableImageEffectBase {

  /**
   * {@inheritdoc}
   */
  public function applyEffect(ImageInterface $image) {
    // This effect doesn't actually modify the local file
    // It's used to mark that this image style should use Thumbor
    // The actual URL generation happens elsewhere via hook_file_url_alter()

    // Log that Thumbor processing was requested for this image style
    $this->logger->info('Thumbor processing configured for image style: @style', [
      '@style' => $this->configuration['image_style'] ?? 'unknown',
    ]);

    // Return TRUE to indicate that the effect was applied successfully
    // (even though no actual image modification occurred)
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, \Drupal\Core\Form\FormStateInterface $form_state) {
    $form['thumbor_url'] = [
      '#type' => 'url',
      '#title' => $this->t('Thumbor Server URL'),
      '#description' => $this->t('Enter the URL to your Thumbor server (e.g., http://your-thumbor-server.com).'),
      '#default_value' => $this->configuration['thumbor_url'],
      '#required' => TRUE,
    ];

    $form['security_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Security Key'),
      '#description' => $this->t('Enter the security key for your Thumbor server (if enabled).'),
      '#default_value' => $this->configuration['security_key'],
      '#required' => FALSE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, \Drupal\Core\Form\FormStateInterface $form_state) {
    parent::submitConfigurationForm($form, $form_state);

    $this->configuration['thumbor_url'] = $form_state->getValue('thumbor_url');
    $this->configuration['security_key'] = $form_state->getValue('security_key');
  }

/**
   * {@inheritdoc}
   */
  public function transformDimensions(array &$dimensions, $uri) {
    // Thumbor effects don't change dimensions since the actual processing
    // happens remotely via URL generation. We don't modify the dimensions.
    // The remote Thumbor service will handle the actual transformations.

    // INIZIO CORREZIONE:
    // Preveniamo l'errore "Division by zero" nella UI di amministrazione.
    // Se le dimensioni in ingresso sono 0 (perché è l'unico effetto),
    // le impostiamo a 1 per evitare un crash matematico.
    if (empty($dimensions['width'])) {
      $dimensions['width'] = 1;
    }
    if (empty($dimensions['height'])) {
      $dimensions['height'] = 1;
    }
    // FINE CORREZIONE
  }

  /**
   * {@inheritdoc}
   */
  public function getDerivativeExtension($extension) {
    // Don't change the extension - Thumbor will handle format conversion remotely
    return $extension;
  }

  /**
   * {@inheritdoc}
   */
  public function getSummary() {
    $summary = [
      '#markup' => $this->t('Process image through Thumbor service'),
      '#settings' => [
        'thumbor_url' => $this->configuration['thumbor_url'],
      ],
    ];

    if (!empty($this->configuration['security_key'])) {
      $summary['#settings']['security_key'] = '***';
    }

    return $summary;
  }

}