<?php

namespace Drupal\imageapi_optimize_thumbor\Service;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\image\Entity\ImageStyle;
use Drupal\Core\File\FileSystemInterface;
use Drupal\imageapi_optimize_thumbor\Plugin\ImageAPIOptimizeProcessor\Thumbor as ThumborProcessor;

/**
 * Service to handle Thumbor URL generation for image styles.
 */
class ThumborService {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * Constructs a new ThumborService object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   * The entity type manager.
   * @param \Drupal\Core\File\FileSystemInterface $file_system
   * The file system service.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, FileSystemInterface $file_system) {
    $this->entityTypeManager = $entity_type_manager;
    $this->fileSystem = $file_system;
  }

  /**
   * Gets the Thumbor URL for a given image and image style.
   *
   * @param string $image_uri
   * The URI of the original image.
   * @param string $image_style_id
   * The image style ID to apply.
   *
   * @return string|false
   * The Thumbor URL for the image with applied style, or FALSE on failure.
   */
  public function getThumborUrl($image_uri, $image_style_id) {
    // Load the image style to check if it uses the Thumbor processor
    $image_style = ImageStyle::load($image_style_id);
    if (!$image_style) {
      // \Drupal::logger('imageapi_optimize_thumbor')->debug('Service: Could not load image style: @style', ['@style' => $image_style_id]);
      return FALSE;
    }

    // Costruisci la stringa delle operazioni (es. "300x200") dagli effetti
    $operations_string = $this->buildThumborOperationsString($image_style);
    // \Drupal::logger('imageapi_optimize_thumbor')->debug('Service: Built operations string: @ops', ['@ops' => $operations_string]);

    // \Drupal::logger('imageapi_optimize_thumbor')->debug('Service: Checking image style @style for Thumbor processors', ['@style' => $image_style_id]);

    // Check for direct Thumbor image effect first
    $effects = $image_style->getEffects();
    foreach ($effects as $effect) {
      // \Drupal::logger('imageapi_optimize_thumbor')->debug('Service: Effect ID found: @id', ['@id' => $effect->getPluginId()]);
      if ($effect->getPluginId() === 'imageapi_optimize_thumbor_effect') {
        // \Drupal::logger('imageapi_optimize_thumbor')->debug('Service: Found direct Thumbor image effect');
        // Get configuration from the image effect
        $config = $effect->getConfiguration();
        $processor_config = $config['data'] ?? [];

        // \Drupal::logger('imageapi_optimize_thumbor')->debug('Service: Direct effect processor config: @config', ['@config' => print_r($processor_config, TRUE)]);

        // Create an instance of the Thumbor processor with the configuration
        $thumbor_processor = new ThumborProcessor(
          $processor_config,
          'thumbor',
          [
            'id' => 'thumbor',
            'label' => t('Thumbor'),
            'description' => t('Provides Thumbor URLs for optimized images without saving files.')
          ],
          \Drupal::logger('imageapi_optimize'),
          \Drupal::service('image.factory'),
          \Drupal::httpClient()
        );

        // Generate and return the Thumbor URL
        $url = $thumbor_processor->getThumborUrl($image_uri, $operations_string);
        // \Drupal::logger('imageapi_optimize_thumbor')->debug('Service: Generated URL from direct effect: @url', ['@url' => $url]);
        return $url;
      }
    }

    // Check if this image style uses ImageAPI Optimize with Thumbor processor
    foreach ($effects as $effect) {
      if ($effect->getPluginId() === 'imageapi_optimize_process') {
        // \Drupal::logger('imageapi_optimize_thumbor')->debug('Service: Found imageapi_optimize_process effect');
        $configuration = $effect->getConfiguration();
        if (isset($configuration['processors'])) {
          foreach ($configuration['processors'] as $processor_id => $processor_config) {
            // \Drupal::logger('imageapi_optimize_thumbor')->debug('Service: Processor ID: @id', ['@id' => $processor_id]);
            if (strpos($processor_id, 'thumbor') !== FALSE) {
              // Create an instance of the Thumbor processor with the configuration
              $processor_data = $processor_config['configuration'] ?? [];
              // \Drupal::logger('imageapi_optimize_thumbor')->debug('Service: ImageAPI Optimize process processor config: @config', ['@config' => print_r($processor_data, TRUE)]);

              $thumbor_processor = new ThumborProcessor(
                $processor_data,
                'thumbor',
                [
                  'id' => 'thumbor',
                  'label' => t('Thumbor'),
                  'description' => t('Provides Thumbor URLs for optimized images without saving files.')
                ],
                \Drupal::logger('imageapi_optimize'),
                \Drupal::service('image.factory'),
                \Drupal::httpClient()
              );

              // Generate and return the Thumbor URL
              $url = $thumbor_processor->getThumborUrl($image_uri, $operations_string);
              // \Drupal::logger('imageapi_optimize_thumbor')->debug('Service: Generated URL from pipeline processor: @url', ['@url' => $url]);
              return $url;
            }
          }
        }
      }
    }

    // Also check if the image style has a pipeline that contains Thumbor processor
    // This is the ImageStyleWithPipeline approach
    if (method_exists($image_style, 'getPipelineEntity')) {
      // \Drupal::logger('imageapi_optimize_thumbor')->debug('Service: Checking pipeline for style @style', ['@style' => $image_style_id]);
      $pipeline = $image_style->getPipelineEntity();
      if ($pipeline) {
        // \Drupal::logger('imageapi_optimize_thumbor')->debug('Service: Pipeline found: @pipeline', ['@pipeline' => $pipeline->id()]);

        // Check the pipeline for Thumbor processor
        $pipeline_processors = $pipeline->getProcessors();
        foreach ($pipeline_processors as $processor) {
          // \Drupal::logger('imageapi_optimize_thumbor')->debug('Service: Pipeline processor ID: @id', ['@id' => $processor->getPluginId()]);
          if (strpos($processor->getPluginId(), 'thumbor') !== FALSE) {
            
            // CORREZIONE: Usa l'istanza del processore esistente
            // \Drupal::logger('imageapi_optimize_thumbor')->debug('Service: Using existing processor instance from pipeline.');

            if ($processor instanceof \Drupal\imageapi_optimize_thumbor\Plugin\ImageAPIOptimizeProcessor\Thumbor) {
              
              // Chiama direttamente il metodo sull'oggetto processore della pipeline
              $url = $processor->getThumborUrl($image_uri, $operations_string);
              
              // \Drupal::logger('imageapi_optimize_thumbor')->debug('Service: Generated URL from associated pipeline: @url', ['@url' => $url]);
              return $url;
            } 
            else {
              // \Drupal::logger('imageapi_optimize_thumbor')->error('Service: Found Thumbor processor, but it is not an instance of ThumborProcessor.');
            }
          }
        }
      }
    }

    // \Drupal::logger('imageapi_optimize_thumbor')->debug('Service: No Thumbor processor found in style @style', ['@style' => $image_style_id]);
    return FALSE;
  }
/**
   * Generates Thumbor operations string from image style effects.
   *
   * @param \Drupal\image\Entity\ImageStyle $image_style
   * The image style entity.
   *
   * @return string
   * A string of Thumbor operations (es. "300x200/filters:format(webp)").
   */
  protected function buildThumborOperationsString(ImageStyle $image_style) {
    $resize_ops = [];
    $filter_ops = [];

    foreach ($image_style->getEffects() as $effect) {
      // Alcuni effetti potrebbero non avere 'data' (es. il nostro effetto thumbor)
      $effect_config = $effect->getConfiguration()['data'] ?? NULL;
      if (!$effect_config) {
          continue;
      }

      switch ($effect->getPluginId()) {
        case 'image_scale':
        case 'image_resize':
          // Formato: [larghezza]x[altezza]
          // Se larghezza o altezza sono nulle, usa 0 (Thumbor capisce)
          $width = $effect_config['width'] ?? 0;
          $height = $effect_config['height'] ?? 0;
          $resize_ops[] = "fit-in/{$width}x{$height}";
          break;

        case 'image_scale_and_crop':
          // Formato: [larghezza]x[altezza]
          $width = $effect_config['width'];
          $height = $effect_config['height'];
          $resize_ops[] = "{$width}x{$height}";
          break;
        
        case 'image_convert':
          if ($effect_config['extension'] === 'webp') {
            $filter_ops[] = 'format(webp)';
          }
          break;
      }
    }

    // Costruisci la stringa finale
    $final_ops = [];
    
    // Aggiungi le operazioni di resize (es. "300x200")
    // Usa solo l'ultima operazione di resize valida trovata
    if (!empty($resize_ops)) {
      $final_ops[] = end($resize_ops); 
    }

    // Aggiungi i filtri (es. "filters:format(webp)")
    if (!empty($filter_ops)) {
      // I filtri multipli sono separati da ":"
      $final_ops[] = 'filters:' . implode(':', $filter_ops);
    }

    // L'URL finale sarà "300x200/filters:format(webp)"
    return implode('/', $final_ops);
  }

  /**
   * Gets the configuration for the Thumbor processor associated with the image style.
   *
   * @param string $image_style_id
   * The image style ID.
   *
   * @return array|false
   * The Thumbor processor configuration, or FALSE if not found.
   */
  protected function getConfigForThumborProcessor($image_style_id) {
    // Load all image styles to find the one that contains Thumbor processor
    $config_factory = \Drupal::configFactory();
    $config_name = 'image.style.' . $image_style_id;
    $config = $config_factory->get($config_name);

    if (!$config->get()) {
      return FALSE;
    }

    $effects = $config->get('effects') ?: [];

    foreach ($effects as $effect) {
      if (isset($effect['id']) && $effect['id'] === 'imageapi_optimize_process') {
        $processors = $effect['data']['processors'] ?? [];

        foreach ($processors as $processor_id => $processor_config) {
          if (strpos($processor_id, 'thumbor') !== FALSE) {
            return $processor_config['configuration'] ?? [];
          }
        }
      }
    }

    return FALSE;
  }
}