<?php

namespace Drupal\images_import\Services;

use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Service for processing a image.
 */
class BatchProcessor {
  use StringTranslationTrait;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The image Processor service.
   *
   * @var Drupal\images_import\Services\ImageProcessor
   */
  protected $imageProcessor;

  /**
   * Constructs a ProcessSingleImage object.
   */
  public function __construct(
    MessengerInterface $messenger,
    ImageProcessor $image_processor,
  ) {
    $this->messenger = $messenger;
    $this->imageProcessor = $image_processor;
  }

  /**
   * Import a single image and assign to entity.
   */
  protected function importImage($row, $images_options) {
    if (empty($row['image_url'])) {
      return FALSE;
    }

    try {
      // Download the image from external URL.
      if ($images_options['is_multiple'] && strpos($row['image_url'], ',') !== FALSE) {
        $image_urls = array_map('trim', explode(',', $row['image_url']));
      }
      else {
        $image_urls = [trim($row['image_url'])];
      }
      $image_count = 0;
      foreach ($image_urls as $image_url) {
        try {
          $image_processed = $this->imageProcessor->processImage($row, $image_url, $images_options);
          if ($image_processed) {
            $image_count++;
          }
        }
        catch (\Exception $e) {
          $this->messenger->addWarning($this->t('Failed to process image from URL: @url - @error', [
            '@url' => $image_url,
            '@error' => $e->getMessage(),
          ]));
          // Continue processing other images instead of throwing.
        }
      }

      if ($image_count === 0) {
        throw new \Exception("No images were successfully processed from the provided URLs for " . $row['entity_title']);
      }
      return TRUE;
    }
    catch (\Exception $e) {
      $this->messenger->addError($this->t('Error importing image: @error', [
        '@error' => $e->getMessage(),
      ]));
      return FALSE;
    }
  }

  /**
   * Batch process callback for image import.
   */
  public static function processItem($row, $images_options, array &$context) {
    // Ensure results array is initialized.
    if (!isset($context['results']) || !is_array($context['results'])) {
      $context['results'] = [
        'count' => 0,
        'error_count' => 0,
      ];
    }

    $form_instance = new static(
      \Drupal::messenger(),
      \Drupal::service('images_import.imageProcessor'),
    );

    if ($images_options['operation'] === "import") {
      $success = $form_instance->importImage($row, $images_options);
    }
    elseif ($images_options['operation'] === "remove") {
      $success = \Drupal::service('images_import.removeImage')->removeImage($row, $images_options);
    }

    if ($success) {
      isset($context['results']['count']) ? $context['results']['count']++ : $context['results']['count'] = 1;
    }
    else {
      isset($context['results']['error_count']) ? $context['results']['error_count']++ : $context['results']['error_count'] = 1;
    }
    $context['message'] = t('Processing image @image_url', ['@image_url' => $row['image_url']]);
  }

  /**
   * Batch finished callback.
   */
  public static function importFinishedCallback($success, array $results, array $operations) {
    $messenger = \Drupal::messenger();
    $format_plural = \Drupal::translation();
    if ($success) {
      if (is_array($results) && isset($results['count'])) {
        $messenger->addMessage($format_plural->formatPlural(
          $results['count'],
          t('Successfully imported images in 1 entity.'),
          t('Successfully imported images in @count entities.')
        ));
      }
      if (is_array($results) && isset($results['error_count'])) {
        $messenger->addError($format_plural->formatPlural(
          $results['error_count'],
          t('Failed to import images in 1 entity.'),
          t('Failed to import images in @count entities.')
        ));
      }
    }
    else {
      $messenger->addError(t('Image import encountered errors.'));
    }
  }

  /**
   * Batch finished callback.
   */
  public static function removeFinishedCallback($success, array $results, array $operations) {
    $messenger = \Drupal::messenger();
    $format_plural = \Drupal::translation();
    if ($success) {
      if (is_array($results) && isset($results['count'])) {
        $messenger->addMessage($format_plural->formatPlural(
          $results['count'],
          t('Successfully removed images from 1 entity.'),
          t('Successfully removed images from @count entities.')
        ));
      }
      if (is_array($results) && isset($results['error_count'])) {
        $messenger->addError($format_plural->formatPlural(
          $results['error_count'],
          t('Failed to remove images from 1 entity.'),
          t('Failed to remove images from @count entities.')
        ));
      }
    }
    else {
      $messenger->addError(t('Image removal encountered errors.'));
    }
  }

}
