<?php

namespace Drupal\images_import\Services;

use Drupal\media\Entity\Media;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\file\FileRepositoryInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;

/**
 * Service for processing a image.
 */
class ImageProcessor {
  use StringTranslationTrait;

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The file repository service.
   *
   * @var \Drupal\file\FileRepositoryInterface
   */
  protected $fileRepository;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The logger service.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $logger;

  /**
   * Constructs a ProcessSingleImage object.
   */
  public function __construct(
    FileSystemInterface $file_system,
    EntityTypeManagerInterface $entity_type_manager,
    FileRepositoryInterface $file_repository,
    MessengerInterface $messenger,
    LoggerChannelFactoryInterface $logger_factory,
  ) {
    $this->fileSystem = $file_system;
    $this->entityTypeManager = $entity_type_manager;
    $this->fileRepository = $file_repository;
    $this->messenger = $messenger;
    $this->logger = $logger_factory;
  }

  /**
   * Processes a single image for import and attaches it to an entity.
   *
   * Downloads the image from the provided URL, saves it to the file system,
   * and assigns it to the specified image field of the given entity.
   *
   * @param array $row
   *   The data row containing information about the image to import.
   * @param string $image_url
   *   The URL of the image to be imported.
   * @param array $images_options
   *   An array of options for the image import process, including:
   *   - entity_type: The type of entity to which the image will be attached.
   *   - bundle: The bundle of the entity (e.g., content type for nodes).
   *   - image_field: The machine name of the image field on the entity.
   *   - entity: The entity object to which the image will be attached.
   *
   * @return bool
   *   TRUE if the image was successfully processed and attached,
   *   FALSE otherwise.
   */
  public function processImage($row, $image_url, $images_options) {
    $entity_type = $images_options['entity_type'];
    $bundle = $images_options['bundle'] ?? NULL;

    // Download the image data.
    $image_data = file_get_contents($image_url);
    if ($image_data === FALSE) {
      $this->messenger->addError($this->t('Failed to download image from @url', ['@url' => $image_url]));
      return FALSE;
    }

    // Create file directory if it doesn't exist.
    $directory = 'public://imported_images/' . $entity_type . '/' . ($bundle ? $bundle . '/' : '');
    $this->fileSystem->prepareDirectory($directory, FileSystemInterface::CREATE_DIRECTORY);

    // Save the file.
    $filename = pathinfo(parse_url($image_url, PHP_URL_PATH), PATHINFO_FILENAME);
    $filename = preg_replace('/[^A-Za-z0-9\-_\.]/', '_', $filename);

    // Try to detect extension from image_url.
    $ext = pathinfo(parse_url($image_url, PHP_URL_PATH), PATHINFO_EXTENSION);
    // Ensure the extension is a valid image extension, fallback to jpg if not.
    $valid_extensions = ['jpg', 'jpeg', 'png', 'gif', 'webp'];
    $ext = strtolower($ext);
    if (!in_array($ext, $valid_extensions)) {
      $this->messenger->addError($this->t('No valid extensions found for image @image. Allowed extensions are: jpg, jpeg, png, gif, webp.', [
        '@image' => $filename,
      ]));
      return FALSE;
    }

    // Add a unique suffix to avoid filename collisions.
    $file_uri = $directory . $filename . '_' . uniqid() . '.' . $ext;

    $file = $this->fileRepository->writeData($image_data, $file_uri, FileSystemInterface::EXISTS_RENAME);
    if ($file) {
      // Set file as permanent.
      $file->setPermanent();
      $file->save();

      // Try to load entity by entity_id if  entity_id available.
      if (isset($row['entity_id'])) {
        $entity_id = $row['entity_id'];
        $entity = $this->entityTypeManager->getStorage($entity_type)->load($entity_id);
        if (!$entity) {
          $this->messenger->addError($this->t('Entity with ID @id not found for image @image', [
            '@id' => $entity_id,
            '@image' => $filename,
          ]));
          return FALSE;
        }
      }
      elseif (isset($row['entity_title'])) {
        if ($entity_type === 'user') {
          $entities = $this->entityTypeManager->getStorage('user')->loadByProperties([
            'name' => $row['entity_title'],
          ]);
        }
        elseif ($entity_type === 'block_content') {
          $entities = $this->entityTypeManager->getStorage('block_content')->loadByProperties([
            'type' => $bundle,
            'info' => $row['entity_title'],
          ]);
        }
        elseif ($entity_type === 'taxonomy_term') {
          $entities = $this->entityTypeManager->getStorage('taxonomy_term')->loadByProperties([
            'vid' => $bundle,
            'name' => $row['entity_title'],
          ]);
        }
        else {
          $entities = $this->entityTypeManager->getStorage($entity_type)->loadByProperties([
            'type' => $bundle,
            'title' => $row['entity_title'],
          ]);
        }

        if (empty($entities)) {
          $this->logger->get('images_import')->info($this->t('Entity with title @title not found for image @image', [
            '@title' => $row['entity_title'],
            '@image' => $filename,
          ]));
          return FALSE;
        }
        $entity = reset($entities);
      }

      // If we have an entity, assign the image to it.
      if ($entity) {
        $this->assignImageToEntity($entity, $images_options, $file);
        $this->logger->get('images_import')->info($this->t('Imported image @filename to @entity_type @entity_id', [
          '@filename' => $filename,
          '@entity_type' => $entity_type,
          '@entity_id' => $entity->id(),
        ]));
      }
      else {
        // Just save the file without assigning to entity.
        $this->logger->get('images_import')->info($this->t('Imported image: @filename (not assigned to any entity)', [
          '@filename' => $filename,
        ]));
      }

      return TRUE;
    }

  }

  /**
   * Assign image to entity field.
   */
  protected function assignImageToEntity(EntityInterface $entity, $images_options, $file) {
    $field_name = $images_options['image_field'];
    $is_multiple = $images_options['is_multiple'];

    // Check if the field exists on the entity.
    if (!$entity->hasField($field_name)) {
      $this->messenger->addError($this->t('Field @field does not exist on entity @id', [
        '@field' => $field_name,
        '@id' => $entity->id(),
      ]));
      return FALSE;
    }

    // Handle media fields.
    $media = NULL;
    $field_definition = $entity->getFieldDefinition($field_name);
    if ($field_definition->getType() === 'entity_reference' && $field_definition->getSetting('target_type') === 'media') {
      // Create a media entity of type "image".
      $media = $this->entityTypeManager->getStorage('media')->create([
        'bundle' => 'image',
        'name' => $file->getFilename(),
        'field_media_image' => [
          'target_id' => $file->id(),
          'alt' => $file->getFilename(),
          'title' => $file->getFilename(),
        ],
        'status' => 1,
      ]);

      $media->save();
    }

    // Get current field values.
    $current_values = $entity->get($field_name)->getValue();

    // Add new image to the field.
    if (!empty($media) && $media instanceof Media) {
      // For media reference fields, set only target_id.
      $new_value = [
        'target_id' => $media->id(),
      ];
    }
    else {
      $new_value = [
        'target_id' => $file->id(),
        'alt' => $file->getFilename(),
        'title' => $file->getFilename(),
      ];
    }

    // For multi-value fields, append the new image.
    if ($is_multiple) {
      $current_values[] = $new_value;
    }
    else {
      // For single-value fields, replace with the new image.
      $current_values = [$new_value];
    }

    // Set the field values.
    $entity->set($field_name, $current_values);

    // Save the entity.
    try {
      $entity->save();
      return TRUE;
    }
    catch (\Exception $e) {
      $this->messenger->addError($this->t('Error saving entity @id: @error', [
        '@id' => $entity->id(),
        '@error' => $e->getMessage(),
      ]));
      return FALSE;
    }
  }

}
